package ru.yandex.persqueue.read.impl.actor;

import java.time.Duration;
import java.util.Map;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import com.google.protobuf.ByteString;
import com.google.protobuf.UnsafeByteOperations;
import com.yandex.ydb.core.Status;
import com.yandex.ydb.core.StatusCode;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;

import ru.yandex.persqueue.codec.Codec;
import ru.yandex.persqueue.read.EventSubscriber;
import ru.yandex.persqueue.read.PartitionStream;
import ru.yandex.persqueue.read.event.CommitAcknowledgementEvent;
import ru.yandex.persqueue.read.event.Message;
import ru.yandex.persqueue.read.event.PartitionStreamClosedEvent;
import ru.yandex.persqueue.read.event.PartitionStreamDestroyEvent;
import ru.yandex.persqueue.read.event.PartitionStreamStatusEvent;
import ru.yandex.persqueue.read.impl.actor.ActorEvents.Disconnect;
import ru.yandex.persqueue.read.impl.actor.ActorEvents.MemoryChunkConsumed;
import ru.yandex.persqueue.read.settings.EventHandlersSettings;
import ru.yandex.persqueue.read.settings.ReadSessionSettings;
import ru.yandex.persqueue.read.settings.RetrySettings;
import ru.yandex.persqueue.read.settings.TopicReadSettings;
import ru.yandex.persqueue.rpc.PqRpcStub;
import ru.yandex.persqueue.rpc.RpcPoolStub;

import static com.yandex.ydb.persqueue.YdbPersqueueV1.Codec.CODEC_RAW;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

/**
 * @author Vladimir Gordiychuk
 */
public class ReadSessionActorImplTest {
    private static final int MAX_MEMORY_USAGE = 5 << 20; // 5 Mib

    @Rule
    public TestName name = new TestName();
    @Rule
    public Timeout timeout = Timeout.builder()
            .withTimeout(10, TimeUnit.SECONDS)
            .build();

    private ReadSessionActor session;
    private EventSubscriber subscriber;
    private RpcPoolStub rpcPool;
    private PqRpcStub rpc;

    @Before
    public void setUp() {
        rpcPool = new RpcPoolStub();
        rpc = rpcPool.getRpc(name.getMethodName());
        subscriber = new EventSubscriber();
        var settings = ReadSessionSettings.newBuilder()
                .addTopic(TopicReadSettings.newBuilder()
                        .path("/topic/alice")
                        .build())
                .addTopic(TopicReadSettings.newBuilder()
                        .path("/topic/bob")
                        .build())
                .consumerName(name.getMethodName() + "/consumer")
                .maxMemoryUsage(MAX_MEMORY_USAGE)
                .executor(ForkJoinPool.commonPool())
                .eventHandlers(EventHandlersSettings.newBuilder()
                        .executor(ForkJoinPool.commonPool())
                        .commonHandler(subscriber)
                        .build())
                .retrySettings(RetrySettings.newBuilder()
                        .backoffSlot(Duration.ofMillis(5))
                        .build())
                .build();
        var publisher = new EventPublisher(new EventProducer() {
            @Override
            public void chunkConsumed(String cluster) {
                session.send(MemoryChunkConsumed.INSTANCE);
            }

            @Override
            public void cancel() {
                session.send(new Disconnect());
            }
        }, settings.handler.executor, ReadSessionActorImpl.MAX_BATCH_SIZE, settings.handler.commonEvents);
        session = new ReadSessionActorImpl("test", name.getMethodName(), settings, publisher, rpcPool);
        session.send(new ActorEvents.Connect());
    }

    @After
    public void tearDown() {
        session.send(new ActorEvents.Disconnect());
    }

    @Test
    public void initRequested() {
        var server = serverSession();
        server.expectInit();
        server.sendInit();
    }

    @Test
    public void initFailed() {
        var server = serverSession();
        server.expectInit();
        server.sendError(Status.of(StatusCode.UNAUTHORIZED));
        server.expectComplete();
        Throwable error = null;
        try {
            subscriber.doneFuture.join();
        } catch (Throwable e) {
            error = e;
        }
        assertNotNull(error);
    }

    @Test
    public void initCompleteByServer() {
        var server = serverSession();
        server.expectInit();
        server.sendComplete();
        server.expectComplete();
        subscriber.doneFuture.join();
    }

    @Test
    public void assignPartition() {
        var server = serverSession();
        server.expectInit();
        server.sendInit();
        var assignId = server.sendAssign("/topic/alice", 1);

        {
            var event = subscriber.expectAssign(assignId, "test", "/topic/alice", 1);
            event.confirm();
        }

        server.expectConfirmAssign(assignId);
    }

    @Test
    public void assignMultipleTopics() {
        var server = serverSession();
        server.expectInit();
        server.sendInit();
        var aliceAssignId = server.sendAssign("/topic/alice", 1);
        var bobAssignId = server.sendAssign("/topic/bob", 100);

        var aliceAssign = subscriber.expectAssign(aliceAssignId, "test", "/topic/alice", 1);
        var bobAssign = subscriber.expectAssign(bobAssignId, "test", "/topic/bob", 100);

        bobAssign.confirm();
        aliceAssign.confirm();

        server.expectConfirmAssign(bobAssignId);
        server.expectConfirmAssign(aliceAssignId);
    }

    @Test
    public void readFailed() {
        var server = serverSession();
        server.expectInit();
        server.sendInit();
        server.expectAssign("/topic/alice", 0);
        server.sendError(Status.of(StatusCode.BAD_REQUEST));
        server.expectComplete();
        Throwable error = null;
        try {
            subscriber.doneFuture.join();
        } catch (Throwable e) {
            error = e;
        }
        assertNotNull(error);
    }

    @Test
    public void readCompleteByServer() {
        var server = serverSession();
        server.expectInit();
        server.sendInit();
        server.expectAssign("/topic/alice", 0);
        server.sendComplete();
        server.expectComplete();
        subscriber.doneFuture.join();
    }

    @Test
    public void startReadingAfterAssign() {
        var server = serverSession();
        server.expectInit();
        server.sendInit();
        server.expectAssign("/topic/alice", 100);
        server.expectRead();
    }

    @Test
    public void gracefulRelease() {
        var server = serverSession();
        server.expectInit();
        server.sendInit();
        var assignId = server.expectAssign("/topic/bob", 1);
        server.sendRelease(assignId, 123, false);
        var event = subscriber.takeEvent(PartitionStreamDestroyEvent.class);
        var partitionStream = event.getPartitionStream();
        assertEquals(assignId, partitionStream.getAssignId());
        assertEquals(123, event.getCommittedOffset());
        event.confirm();
        server.expectConfirmRelease(assignId);
    }

    @Test
    public void multipleRelease() {
        var server = serverSession();
        server.expectInit();
        server.sendInit();
        var aliceAssignId = server.expectAssign("/topic/alice", 1);
        var bobAssignId = server.expectAssign("/topic/bob", 1);

        server.sendRelease(aliceAssignId, 1, false);
        server.sendRelease(bobAssignId, 2, false);

        {
            var event = subscriber.takeEvent(PartitionStreamDestroyEvent.class);
            var partitionStream = event.getPartitionStream();
            assertEquals(aliceAssignId, partitionStream.getAssignId());
            assertEquals(1, event.getCommittedOffset());
        }

        {
            var event = subscriber.takeEvent(PartitionStreamDestroyEvent.class);
            var partitionStream = event.getPartitionStream();
            assertEquals(bobAssignId, partitionStream.getAssignId());
            assertEquals(2, event.getCommittedOffset());
            event.confirm();
        }

        server.expectConfirmRelease(bobAssignId);
    }

    @Test
    public void partitionStatus() {
        var server = serverSession();
        server.expectInit();
        server.sendInit();
        var topic = "/topic/alice";
        var partition = 100L;

        PartitionStream partitionStream;
        long assignId;
        {
            assignId = server.sendAssign(topic, partition);
            var assign = subscriber.expectAssign(assignId, "test", topic, partition);
            assign.confirm();
            partitionStream = assign.getPartitionStream();
            server.expectConfirmAssign(assignId);
        }

        partitionStream.requestStatus();
        server.expectPartitionStatus(assignId);
        server.sendPartitionStatus(assignId, 1, 2, 3);

        var event = subscriber.takeEvent(PartitionStreamStatusEvent.class);
        assertEquals(partitionStream, event.getPartitionStream());
        assertEquals(1, event.getCommittedOffset());
        assertEquals(2, event.getEndOffset());
        assertEquals(3, event.getWriteWatermark());
    }

    @Test
    public void messageData() {
        var server = serverSession();
        server.expectInit();
        server.sendInit();
        long assignId = server.expectAssign("/topic/alice", 1);
        server.expectRead();

        var expectedMeta = Map.of("myKey", "myValue");
        var messageOne = ByteString.copyFromUtf8("hi");
        var messageTwo = ByteString.copyFromUtf8("bob");

        server.sendData(assignId, 42, expectedMeta, CODEC_RAW, messageOne, messageTwo);
        {
            var event = subscriber.takeEvent(Message.class);
            assertEquals(42, event.getOffset());
            assertEquals(0, event.getSeqNo());
            assertEquals(expectedMeta, event.getExtraFields());
            assertEquals(messageOne, event.getData());
            assertEquals(assignId, event.getPartitionStream().getAssignId());
        }

        {
            var event = subscriber.takeEvent(Message.class);
            assertEquals(43, event.getOffset());
            assertEquals(1, event.getSeqNo());
            assertEquals(expectedMeta, event.getExtraFields());
            assertEquals(messageTwo, event.getData());
            assertEquals(assignId, event.getPartitionStream().getAssignId());
        }
    }

    @Test
    public void commit() {
        var server = serverSession();
        server.expectInit();
        server.sendInit();
        long assignId = server.expectAssign("/topic/alice", 1);
        server.expectRead();

        var expectedMeta = Map.of("myKey", "myValue");
        var messageOne = ByteString.copyFromUtf8("hi");
        var messageTwo = ByteString.copyFromUtf8("bob");

        var cookie = server.sendData(assignId, 42, expectedMeta, CODEC_RAW, messageOne, messageTwo);
        {
            var event = subscriber.takeEvent(Message.class);
            assertEquals(42, event.getOffset());
            assertEquals(0, event.getSeqNo());
            assertEquals(expectedMeta, event.getExtraFields());
            assertEquals(messageOne, event.getData());
            assertEquals(assignId, event.getPartitionStream().getAssignId());
            event.commit();
        }

        {
            var event = subscriber.takeEvent(Message.class);
            assertEquals(43, event.getOffset());
            assertEquals(1, event.getSeqNo());
            assertEquals(expectedMeta, event.getExtraFields());
            assertEquals(messageTwo, event.getData());
            assertEquals(assignId, event.getPartitionStream().getAssignId());
            event.commit();
        }

        server.expectRead();
        server.expectCommit(assignId, cookie);
        assertNull(subscriber.events.poll());

        server.sendCommitted(assignId, cookie);
        {
            var event = subscriber.takeEvent(CommitAcknowledgementEvent.class);
            assertEquals(assignId, event.getPartitionStream().getAssignId());
            assertEquals(43, event.getCommittedOffset());
        }
    }

    @Test
    public void messageFromPreviousAssignIgnored() {
        var server = serverSession();
        server.expectInit();
        server.sendInit();
        long oldAssignId = server.expectAssign("/topic/alice", 1);
        server.expectRead();
        server.expectRelease(oldAssignId);
        long newAssignId = server.expectAssign("/topic/alice", 1);

        server.sendData(oldAssignId, 42, Map.of(), CODEC_RAW, ByteString.copyFromUtf8("to be ignore"));
        server.sendData(newAssignId, 43, Map.of(), CODEC_RAW, ByteString.copyFromUtf8("actual"));

        {
            var event = subscriber.takeEvent(Message.class);
            assertEquals(43, event.getOffset());
            assertEquals(newAssignId, event.getPartitionStream().getAssignId());
            assertEquals(ByteString.copyFromUtf8("actual"), event.getData());
        }
    }

    @Test
    public void stopReadingWhenMemoryLimitReached() {
        subscriber.autoRequestMore = false;

        var server = serverSession();
        server.expectInit();
        server.sendInit();
        long assignId = server.expectAssign("/topic/alice", 1);
        server.expectRead();
        long cookie = server.sendData(assignId, 42, Map.of(), CODEC_RAW, ByteString.EMPTY);

        // stop consuming, because subscriber not took new events
        {
            int alreadyReadBytes = 0;
            long offset = 43;
            var random = ThreadLocalRandom.current();
            while (alreadyReadBytes + ReadSessionActorImpl.MAX_BATCH_SIZE < MAX_MEMORY_USAGE) {
                server.expectRead();
                byte[] data = new byte[random.nextInt(1 << 20)];
                random.nextBytes(data);
                server.sendData(assignId, offset++, Map.of(), CODEC_RAW, UnsafeByteOperations.unsafeWrap(data));
                alreadyReadBytes += data.length;
            }

            // message with empty content can't trigger overload by memory
            {
                subscriber.subscription.request(1);
                var message = subscriber.takeEvent(Message.class);
                assertEquals(ByteString.EMPTY, message.getData());
                message.commit();
                server.expectCommit(assignId, cookie);
                server.sendCommitted(assignId, cookie);
            }

            // active reads available only when consumer at least one megabyte
            server.expectNoReads();
        }

        // continue read when subscribe start consume
        {
            int alreadyReadBytes = 0;
            while (alreadyReadBytes < (MAX_MEMORY_USAGE / 2)) {
                subscriber.subscription.request(1);
                var message = subscriber.takeEvent(Message.class);
                alreadyReadBytes += message.memoryUseBytes();
            }

            server.expectRead();
        }
    }

    @Test
    public void retry() {
        // failed data receive
        Message messageOne;
        {
            var server = serverSession();
            server.expectInit();
            server.sendInit();
            long assignId = server.expectAssign("/topic/alice", 1);
            server.expectRead();
            server.sendData(assignId, 42, Map.of(), CODEC_RAW, ByteString.copyFromUtf8("one"));
            messageOne = subscriber.takeEvent(Message.class);
            server.sendError(Status.of(StatusCode.UNAVAILABLE));
            server.expectComplete();
            var closeEvent = subscriber.takeEvent(PartitionStreamClosedEvent.class);
            assertEquals(messageOne.getPartitionStream(), closeEvent.getPartitionStream());

            // commit should be ignored because no active session
            messageOne.commit();
        }

        // failed init
        {
            var server = serverSession();
            server.expectInit();
            server.sendError(Status.of(StatusCode.UNAVAILABLE));
            server.expectComplete();
        }

        // success init, failed read
        {
            var server = serverSession();
            server.expectInit();
            server.sendInit();
            var assignId = server.expectAssign("/topic/alice", 1);
            server.expectRead();
            server.sendError(Status.of(StatusCode.UNAVAILABLE));
            server.expectComplete();

            var closeEvent = subscriber.takeEvent(PartitionStreamClosedEvent.class);
            assertEquals(assignId, closeEvent.getPartitionStream().getAssignId());
        }

        // success init and read
        Message messageTwo;
        {
            var server = serverSession();
            server.expectInit();
            server.sendInit();
            long assignId = server.expectAssign("/topic/alice", 1);
            server.expectRead();
            server.sendData(assignId, 42, Map.of(), CODEC_RAW, ByteString.copyFromUtf8("one"));
            messageTwo = subscriber.takeEvent(Message.class);
            assertEquals(ByteString.copyFromUtf8("one"), messageTwo.getData());
            assertEquals(Codec.RAW, messageTwo.getCodec());
        }
    }

    public ServerSession serverSession() {
        try {
            while (true) {
                var observer = rpc.getActiveReadObserver();
                if (observer == null) {
                    TimeUnit.MILLISECONDS.sleep(10L);
                    continue;
                }

                return new ServerSession(observer, name.getMethodName(), "test", subscriber);
            }
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

}
