package ru.yandex.persqueue.read.impl.actor;

import java.time.Duration;
import java.util.concurrent.ArrayBlockingQueue;

import com.yandex.ydb.core.StatusCode;
import org.hamcrest.Matchers;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.persqueue.read.impl.actor.ActorEvents.Connect;
import ru.yandex.persqueue.read.settings.RetrySettings;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;

/**
 * @author Vladimir Gordiychuk
 */
public class ReadSessionRetryContextTest {

    private ArrayBlockingQueue<ActorEvent> events;

    @Before
    public void setUp() {
        events = new ArrayBlockingQueue<>(100);
    }

    @Test
    public void unableScheduleRetryInvalidStatus() {
        var context = newContext(RetrySettings.newBuilder()
                .maxRetries(10)
                .build());

        assertFalse(context.scheduleRetry(StatusCode.BAD_REQUEST));
        assertFalse(context.scheduleRetry(StatusCode.UNAUTHORIZED));
        assertFalse(context.scheduleRetry(StatusCode.PRECONDITION_FAILED));
        assertFalse(context.scheduleRetry(StatusCode.UNSUPPORTED));
        assertFalse(context.scheduleRetry(StatusCode.CLIENT_UNAUTHENTICATED));
        assertFalse(context.scheduleRetry(StatusCode.CLIENT_CALL_UNIMPLEMENTED));
    }

    @Test
    public void unableScheduleMaxAttempt() {
        var context = newContext(RetrySettings.newBuilder()
                .maxRetries(0)
                .build());

        assertFalse(context.scheduleRetry(StatusCode.UNAVAILABLE));
        assertFalse(context.scheduleRetry(StatusCode.CLIENT_RESOURCE_EXHAUSTED));
    }

    @Test
    public void unableScheduleReachedMaxAttempt() {
        var context = newContext(RetrySettings.newBuilder()
                .maxRetries(2)
                .backoffSlot(Duration.ofMillis(2))
                .build());

        assertTrue(context.scheduleRetry(StatusCode.UNAVAILABLE));
        assertNotNull(takeEvent());
        assertTrue(context.scheduleRetry(StatusCode.UNAVAILABLE));
        assertNotNull(takeEvent());
        assertFalse(context.scheduleRetry(StatusCode.UNAVAILABLE));

    }

    @Test
    public void scheduleRetryUntilSuccess() {
        var context = newContext(RetrySettings.newBuilder()
                .backoffSlot(Duration.ofMillis(10))
                .backoffCeiling(5)
                .maxRetries(Integer.MAX_VALUE)
                .build());

        // at least 3 times session failed
        for (int index = 0; index < 3; index++) {
            assertTrue(context.scheduleRetry(StatusCode.UNAVAILABLE));
            assertNotNull(takeEvent());
        }

        // mark success
        context.success();
    }

    private ActorEvents.Connect takeEvent() {
        try {
            var event = events.take();
            assertThat(event, Matchers.instanceOf(ActorEvents.Connect.class));
            return (ActorEvents.Connect) event;
        } catch (InterruptedException e) {
            throw new RuntimeException();
        }
    }

    private ReadSessionRetryContext newContext(RetrySettings settings) {
        return new ReadSessionRetryContext(settings, () -> events.add(new Connect()));
    }
}
