package ru.yandex.persqueue.rpc;

import java.util.concurrent.Callable;
import java.util.concurrent.CyclicBarrier;
import java.util.concurrent.ForkJoinPool;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.persqueue.settings.GrpcTransportSettings;

import static org.junit.Assert.assertNotSame;
import static org.junit.Assert.assertSame;

/**
 * @author Vladimir Gordiychuk
 */
public class GrpcRpcPoolTest {
    private GrpcTransportSettings settings;
    private GrpcRpcPool pool;

    @Before
    public void setUp() {
        settings = GrpcTransportSettings.newBuilder()
                .endpoint("localhost")
                .build();
        pool = new GrpcRpcPool(settings);
    }

    @Test
    public void sameRpcForSameEndpoint() {
        try (var left = pool.getRpc("localhost");
             var right = pool.getRpc("localhost"))
        {
            assertSame(left, right);
        }
    }

    @Test
    public void avoidReturnClosed() {
        PqRpc left = pool.getRpc("localhost");
        left.close();

        try (var right = pool.getRpc("localhost")) {
            assertNotSame(left, right);
        }
    }

    @Test
    public void avoidCloseTooEarlier() {
        try (var left = pool.getRpc("localhost"))
        {
            // previous rpc steal in use
            pool.getRpc("localhost").close();
            try (var right = pool.getRpc("localhost")) {
                assertSame(left, right);
            }
        }
    }

    @Test
    public void concurrentGet() {
        CyclicBarrier barrier = new CyclicBarrier(2);
        Callable<PqRpc> create = () -> {
            try {
                barrier.await();
                return pool.getRpc("localhost");
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        };

        for (int index = 0; index < 10; index++) {
            barrier.reset();

            var futureLeft = ForkJoinPool.commonPool().submit(create);
            var futureRight = ForkJoinPool.commonPool().submit(create);
            try (var right = futureRight.join(); var left = futureLeft.join()) {
                assertSame(left, right);
            }
        }
    }

    @Test
    public void concurrentGetAndClose() {
        CyclicBarrier barrier = new CyclicBarrier(2);
        for (int index = 0; index < 10; index++) {
            barrier.reset();

            var init = pool.getRpc("localhost");
            var futureCreate = ForkJoinPool.commonPool().submit(() -> {
                try {
                    barrier.await();
                    return pool.getRpc("localhost");
                } catch (Exception e) {
                    throw new RuntimeException(e);
                }
            });
            var futureClose = ForkJoinPool.commonPool().submit(() -> {
                try {
                    barrier.await();
                    init.close();
                } catch (Exception e) {
                    throw new RuntimeException(e);
                }
            });

            futureClose.join();
            try (var left = futureCreate.join();
                 var right = pool.getRpc("localhost"))
            {
                assertSame(left, right);
            }
        }
    }
}
