package ru.yandex.persqueue.rpc;

import java.util.concurrent.ArrayBlockingQueue;

import javax.annotation.Nullable;

import com.google.common.base.MoreObjects;
import com.google.protobuf.TextFormat;
import com.yandex.ydb.core.rpc.OutStreamObserver;
import com.yandex.ydb.core.rpc.StreamObserver;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadClientMessage;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadServerMessage;
import org.hamcrest.Matchers;

import ru.yandex.persqueue.read.impl.protocol.ClientRequestProto;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;

/**
 * @author Vladimir Gordiychuk
 */
public class OutReadSessionObserverStub implements OutStreamObserver<MigrationStreamingReadClientMessage> {
    public final ArrayBlockingQueue<Event> events = new ArrayBlockingQueue<>(10000);
    public final StreamObserver<MigrationStreamingReadServerMessage> observer;
    public volatile boolean competed;

    public OutReadSessionObserverStub(StreamObserver<MigrationStreamingReadServerMessage> observer) {
        this.observer = observer;
    }

    @Nullable
    public Event pullEvent() {
        return events.poll();
    }

    public Event takeEvent() {
        try {
            return events.take();
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    public void expectComplete() {
        while (true) {
            var event = takeEvent();
            assertNull(event.toString(), event.error);
            if (event.complete) {
                return;
            }
        }
    }

    public void expectError() {
        var event = takeEvent();
        assertNotNull(event.toString(), event.error);
        assertFalse(event.toString(), event.complete);
    }

    public MigrationStreamingReadClientMessage takeMessage() {
        var event = takeEvent();
        assertNotNull(event.toString(), event.message);
        assertNull(event.error);
        assertFalse(event.complete);
        return event.message;
    }

    public <T> T takeRequest(Class<T> clazz) {
        var message = takeMessage();
        var req = ClientRequestProto.getRequest(message);
        assertThat(req, Matchers.instanceOf(clazz));
        return clazz.cast(req);
    }

    @Override
    public void onNext(MigrationStreamingReadClientMessage value) {
        events.offer(new Event(value));
    }

    @Override
    public void onError(Throwable t) {
        events.offer(new Event(t));
    }

    @Override
    public void onCompleted() {
        events.add(new Event(true));
        competed = true;
    }

    public static class Event {
        public MigrationStreamingReadClientMessage message;
        public boolean complete;
        public Throwable error;

        public Event(MigrationStreamingReadClientMessage message) {
            this.message = message;
        }

        public Event(boolean complete) {
            this.complete = complete;
        }

        public Event(Throwable error) {
            this.error = error;
        }

        @Override
        public String toString() {
            return MoreObjects.toStringHelper(this)
                    .omitNullValues()
                    .add("message", message != null ? TextFormat.shortDebugString(message) : null)
                    .add("complete", complete)
                    .add("error", error)
                    .toString();
        }
    }
}
