package ru.yandex.staff;

import java.time.Duration;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;

import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static java.util.Objects.requireNonNull;

/**
 * @author Vladimir Gordiychuk
 */
public class StaffClientOptions {
    private final String url;
    private final String oauthToken;
    private final MetricRegistry metricRegistry;
    private final long cacheTtlMillis;
    private final Duration connectionTimeout;
    private final Executor executor;

    private StaffClientOptions(Builder builder) {
        this.url = builder.url != null
                ? builder.url
                : "https://staff-api.yandex-team.ru";
        this.oauthToken = requireNonNull(builder.oauthToken);
        this.metricRegistry = builder.metricRegistry != null
                ? builder.metricRegistry
                : MetricRegistry.root();
        this.cacheTtlMillis = builder.cacheTtlMillis != 0
                ? builder.cacheTtlMillis
                : TimeUnit.HOURS.toMillis(1L);
        this.executor = requireNonNull(builder.executor);
        this.connectionTimeout = builder.connectionTimeout != null
                ? builder.connectionTimeout
                : Duration.ofSeconds(10);
    }

    public Duration getConnectionTimeout() {
        return connectionTimeout;
    }

    public Executor getExecutor() {
        return executor;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public String getUrl() {
        return url;
    }

    public String getOauthToken() {
        return oauthToken;
    }

    public long getCacheTtlMillis() {
        return cacheTtlMillis;
    }

    public MetricRegistry getMetricRegistry() {
        return metricRegistry;
    }

    public static class Builder {
        @Nullable
        private String url;
        @Nullable
        private String oauthToken;
        @Nullable
        private MetricRegistry metricRegistry;
        private long cacheTtlMillis;
        @Nullable
        private Duration connectionTimeout;
        private Executor executor;

        private Builder() {
        }

        public Builder setConnectionTimeout(Duration duration) {
            this.connectionTimeout = duration;
            return this;
        }

        public Builder setExecutor(Executor executor) {
            this.executor = executor;
            return this;
        }

        public Builder setUrl(String url) {
            this.url = "".equals(url) ? null : url;
            return this;
        }

        public Builder setOauthToken(String oauthToken) {
            this.oauthToken = "".equals(oauthToken) ? null : oauthToken;
            return this;
        }

        public Builder setMetricRegistry(MetricRegistry metricRegistry) {
            this.metricRegistry = metricRegistry;
            return this;
        }

        public Builder setCacheTtlMillis(long cacheTtlMillis) {
            this.cacheTtlMillis = cacheTtlMillis;
            return this;
        }

        public Builder setCacheTtl(long value, TimeUnit unit) {
            this.cacheTtlMillis = unit.toMillis(value);
            return this;
        }

        public StaffClientOptions build() {
            return new StaffClientOptions(this);
        }
    }
}
