package ru.yandex.staff.http;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Strings;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import io.netty.handler.codec.http.HttpHeaderNames;
import io.netty.handler.codec.http.HttpStatusClass;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.staff.StaffClient;
import ru.yandex.staff.StaffClientOptions;
import ru.yandex.staff.StaffGroup;
import ru.yandex.staff.StaffGroupMember;
import ru.yandex.staff.UserInfo;
import ru.yandex.staff.exceptions.StaffClientException;
import ru.yandex.staff.exceptions.UnknownStaffException;

/**
 * @author Vladimir Gordiychuk
 */
public class HttpStaffClient implements StaffClient {

    private static final Logger logger = LoggerFactory.getLogger(HttpStaffClient.class);
    private static final String STAFF_FIELDS =
            "uid,login,work_email,phones.is_main,phones.number,phones.type,phones.kind,official.is_dismissed,accounts,name";
    private static final String STAFF_GROUP_MEMBER_FIELDS = "login";
    private static final String STAFF_GROUP_FIELDS = "id,type,name";
    private static final String GROUP_MEMBER_LIMIT = "1000";
    private static final Duration DEFAULT_REQUEST_TIMEOUT_MILLIS = Duration.ofSeconds(30);
    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
            .withNumRetries(3)
            .withDelay(1_000)
            .withMaxDelay(60_000);

    private final HttpClient httpClient;
    private final StaffClientOptions opts;

    private final HttpStaffClientMetrics metrics;
    private final ObjectMapper mapper = new ObjectMapper();
    private final Cache<String, TimeExpired<UserInfo>> userInfoCache;

    public HttpStaffClient(StaffClientOptions opts) {
        this.httpClient = HttpClient.newBuilder()
                .version(HttpClient.Version.HTTP_1_1)
                .followRedirects(HttpClient.Redirect.NEVER)
                .connectTimeout(opts.getConnectionTimeout())
                .executor(opts.getExecutor())
                .build();
        this.opts = opts;

        this.metrics = new HttpStaffClientMetrics(opts.getMetricRegistry());
        this.userInfoCache = CacheBuilder.newBuilder()
                .expireAfterAccess(opts.getCacheTtlMillis(), TimeUnit.MILLISECONDS)
                .build();
    }

    @Override
    public CompletableFuture<UserInfo> getUserInfo(String login) {
        TimeExpired<UserInfo> cached = userInfoCache.getIfPresent(login);
        if (cached == null) {
            return execPersonRequest(login);
        }

        if (cached.isFresh(opts.getCacheTtlMillis())) {
            return CompletableFuture.completedFuture(cached.value);
        }

        return execPersonRequest(login)
                .exceptionally(e -> {
                    logger.error("Refreshing UseInfo failed, use obsoleted data by login {}", login, e);
                    return cached.value;
                });
    }

    private CompletableFuture<String> makeHttpRequestAndExpectSuccess(String uri, HttpStaffClientMetrics.Endpoint endpoint) {
        var request = HttpRequest.newBuilder(URI.create(uri))
                .header(HttpHeaderNames.AUTHORIZATION.toString(), "OAuth " + opts.getOauthToken())
                .timeout(DEFAULT_REQUEST_TIMEOUT_MILLIS)
                .build();
        long startNanos = System.nanoTime();
        endpoint.started();
        return httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString())
                .whenComplete((response, throwable) -> {
                    endpoint.complete(System.nanoTime() - startNanos);
                    if (throwable == null) {
                        endpoint.status(response.statusCode());
                        ensureHttpStatusSuccess(response);
                    }
                })
                .thenApply(HttpResponse::body);
    }

    private void ensureHttpStatusSuccess(HttpResponse<String> response) {
        if (!HttpStatusClass.SUCCESS.contains(response.statusCode())) {
            logger.error("Staff response (status {}):\n{}, body {}", response.statusCode(), response, response.body());
            throw StaffClientException.fromHttpResponse(response);
        }
    }

    @Override
    public CompletableFuture<List<UserInfo>> getUserInfo(List<String> login) {
        if (login.isEmpty()) {
            return CompletableFuture.completedFuture(List.of());
        }
        HttpStaffClientMetrics.Endpoint endpoint = metrics.getEndpoint("/v3/persons");
        String uri = opts.getUrl() + "/v3/persons" +
                "?_limit=" + login.size() +
                "&login=" + String.join(",", login) +
                "&_fields=" + STAFF_FIELDS;
        return makeHttpRequestAndExpectSuccess(uri, endpoint)
                .thenApply(this::parsePersonsResponse);
    }

    @Override
    public CompletableFuture<UserInfo> getUserInfoByTelegramLogin(String telegramLogin) {
        HttpStaffClientMetrics.Endpoint endpoint = metrics.getEndpoint("/v3/persons");
        String uri = opts.getUrl() + "/v3/persons" +
                "?accounts.value_lower=" + telegramLogin.toLowerCase(Locale.ROOT) +
                "&_one=1" +
                "&_fields=" + STAFF_FIELDS;
        return makeHttpRequestAndExpectSuccess(uri, endpoint)
                .thenApply(this::parsePersonResponse);
    }

    @Override
    public CompletableFuture<List<StaffGroupMember>> getStaffGroupMembers(String groupId) {
        return new GroupsLoader(groupId).load()
                .thenCompose((groups) -> CompletableFutures.safeCall(() -> {
                    List<StaffGroupMember> result = new ArrayList<>();
                    CompletableFuture<Void> resultFuture = CompletableFuture.completedFuture(null);
                    for (var group : groups) {
                        resultFuture = resultFuture
                                .thenCompose(unused -> new GroupMembersLoader(group).load())
                                .thenApply(staffGroupMembers -> {
                                    result.addAll(staffGroupMembers);
                                    return null;
                                });
                    }
                    return resultFuture.thenApply(unused -> result);
                }));
    }

    @Override
    public CompletableFuture<List<StaffGroup>> getStaffGroup(List<String> groupIds) {
        if (groupIds.isEmpty()) {
            return CompletableFuture.completedFuture(List.of());
        }
        HttpStaffClientMetrics.Endpoint endpoint = metrics.getEndpoint("/v3/groups");
        String uri = opts.getUrl() + "/v3/groups" +
                "?_limit=" + groupIds.size() +
                "&id=" + String.join(",", groupIds) +
                "&_fields=" + STAFF_GROUP_FIELDS;
        return makeHttpRequestAndExpectSuccess(uri, endpoint)
                .thenApply(this::parseGroupsResponse);
    }

    private CompletableFuture<UserInfo> execPersonRequest(String login) {
        HttpStaffClientMetrics.Endpoint endpoint = metrics.getEndpoint("/v3/persons");
        String uri = opts.getUrl() + "/v3/persons" +
                "?login=" + login +
                "&_one=1" +
                "&_fields=" + STAFF_FIELDS;
        return makeHttpRequestAndExpectSuccess(uri, endpoint)
                .thenApply(response -> {
                    UserInfo info = parsePersonResponse(response);
                    userInfoCache.put(info.getLogin(), new TimeExpired<>(System.currentTimeMillis(), info));
                    return info;
                });
    }


    private UserInfo parsePersonResponse(String body) {
        try {
            StaffPersonResponse response = mapper.readValue(body, StaffPersonResponse.class);
            return getUserInfo(response);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private UserInfo getUserInfo(StaffPersonResponse response) {
        List<String> telegramLogins = response.accounts.stream()
                .filter(e -> "telegram".equals(e.type))
                .filter(e -> !e.isPrivate)
                .map(e -> e.value)
                .collect(Collectors.toList());
        var firstName = Optional.ofNullable(response.name)
                .map(name -> name.first)
                .map(multilanguageString -> multilanguageString.ru)
                .orElse("");
        var lastName = Optional.ofNullable(response.name)
                .map(name -> name.last)
                .map(multilanguageString -> multilanguageString.ru)
                .orElse("");

        return new UserInfo(response.uid, response.login, findPhone(response), response.official.dismissed,
                telegramLogins, firstName, lastName);
    }

    private List<StaffGroup> parseGroupsResponse(String body) {
        try {
            StaffGroupResponse response = mapper.readValue(body, StaffGroupResponse.class);
            return response.groups;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private List<UserInfo> parsePersonsResponse(String body) {
        try {
            StaffPersonsResponse response = mapper.readValue(body, StaffPersonsResponse.class);
            List<UserInfo> userInfo = new ArrayList<>(response.persons.size());
            for (StaffPersonResponse person : response.persons) {
                userInfo.add(getUserInfo(person));
            }
            return userInfo;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Nullable
    private String findPhone(StaffPersonResponse response) {
        @Nullable
        StaffPersonResponse.Phone candidate = null;
        for (StaffPersonResponse.Phone phone : response.phones) {
            if (!"mobile".equals(phone.type)) {
                continue;
            }

            if (phone.main) {
                candidate = phone;
            }

            if ("monitoring".equals(phone.kind)) {
                candidate = phone;
                break;
            }
        }

        if (candidate != null) {
            return candidate.number;
        }

        return null;
    }

    @Override
    public void close() {
    }

    private static final class TimeExpired<T> {
        final long createdAt;
        final T value;

        TimeExpired(long createdAt, T value) {
            this.createdAt = createdAt;
            this.value = value;
        }

        private boolean isFresh(long ttl) {
            return System.currentTimeMillis() - createdAt <= ttl;
        }
    }

    private final class GroupMembersLoader {
        private final String groupId;
        private Integer currentPage = 1;
        private List<StaffGroupMember> result = new ArrayList<>();
        private final CompletableFuture<List<StaffGroupMember>> doneFuture = new CompletableFuture<>();

        public GroupMembersLoader(String groupId) {
            this.groupId = groupId;
        }

        public CompletableFuture<List<StaffGroupMember>> load() {
            loadNextPage();
            return doneFuture;
        }

        private void loadNextPage() {
            RetryCompletableFuture.runWithRetries(this::request, RETRY_CONFIG)
                    .whenComplete(this::onPageLoad);
        }

        private CompletableFuture<StaffGroupMembersResponse> request() {
            HttpStaffClientMetrics.Endpoint endpoint = metrics.getEndpoint("/v3/persons");
            String uri = opts.getUrl() + "/v3/persons" +
                    "?is_deleted=false" +
                    "&official.is_dismissed=false" +
                    "&groups.group.id=" + groupId +
                    "&_limit=" + GROUP_MEMBER_LIMIT +
                    "&_page=" + currentPage +
                    "&_fields=" + STAFF_GROUP_MEMBER_FIELDS;
            return makeHttpRequestAndExpectSuccess(uri, endpoint)
                    .thenApply(response -> {
                        try {
                            return mapper.readValue(response, StaffGroupMembersResponse.class);
                        } catch (IOException e) {
                            throw new RuntimeException(e);
                        }
                    });
        }

        private void onPageLoad(StaffGroupMembersResponse resp, @Nullable Throwable e) {
            if (e != null) {
                doneFuture.completeExceptionally(e);
                return;
            }

            if (!Strings.isNullOrEmpty(resp.errorMessage)) {
                doneFuture.completeExceptionally(new UnknownStaffException(resp.errorMessage));
                return;
            }

            result.addAll(resp.members);
            if (resp.isLast()) {
                doneFuture.complete(result);
            } else {
                currentPage++;
                loadNextPage();
            }
        }
    }

    private final class GroupsLoader {
        private final String startGroupId;
        private List<String> result = new ArrayList<>();
        private Set<String> toLoad = new LinkedHashSet<>();
        private final CompletableFuture<List<String>> doneFuture = new CompletableFuture<>();

        public GroupsLoader(String groupId) {
            this.startGroupId = groupId;
            toLoad.add(startGroupId);
        }

        public CompletableFuture<List<String>> load() {
            loadNextPage(startGroupId);
            return doneFuture;
        }

        private void loadNextPage(String groupId) {
            RetryCompletableFuture.runWithRetries(() -> request(groupId), RETRY_CONFIG)
                    .whenComplete(this::onPageLoad);
        }

        private CompletableFuture<StaffGroupsResponse> request(String groupId) {
            HttpStaffClientMetrics.Endpoint endpoint = metrics.getEndpoint("/v3/groups");
            String uri = opts.getUrl() + "/v3/groups" +
                    "?parent.id=" + groupId +
                    "&_limit=1000";
            return makeHttpRequestAndExpectSuccess(uri, endpoint)
                    .thenApply(response -> {
                        try {
                            result.add(groupId);
                            toLoad.remove(groupId);
                            return mapper.readValue(response, StaffGroupsResponse.class);
                        } catch (IOException e) {
                            throw new RuntimeException(e);
                        }
                    });
        }

        private void onPageLoad(StaffGroupsResponse resp, @Nullable Throwable e) {
            if (e != null) {
                doneFuture.completeExceptionally(e);
                return;
            }

            if (!Strings.isNullOrEmpty(resp.errorMessage)) {
                doneFuture.completeExceptionally(new UnknownStaffException(resp.errorMessage));
                return;
            }
            var groups = resp.groups == null
                    ? List.<String>of()
                    : resp.groups.stream().map(staffGroup -> String.valueOf(staffGroup.id)).collect(Collectors.toList());
            toLoad.addAll(groups);
            if (toLoad.isEmpty()) {
                doneFuture.complete(result);
            } else {
                loadNextPage(toLoad.iterator().next());
            }
        }
    }

}
