package ru.yandex.staff.http;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.LazyGaugeInt64;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class HttpStaffClientMetrics {
    private final MetricRegistry registry;
    private final ConcurrentMap<String, Endpoint> endpoints = new ConcurrentHashMap<>();

    public HttpStaffClientMetrics(MetricRegistry registry) {
        this.registry = registry;
    }

    public Endpoint getEndpoint(String endpoint) {
        return endpoints.computeIfAbsent(endpoint, s -> new Endpoint(registry, s));
    }

    public static class Endpoint {
        private final MetricRegistry registry;

        private final Rate started;
        private final Rate completed;
        private final LazyGaugeInt64 inFlight;
        private final Histogram elapsedTimeMillis;
        private final ConcurrentMap<Integer, Rate> statusCodes = new ConcurrentHashMap<>();

        public Endpoint(MetricRegistry registry, String endpoint) {
            this(registry.subRegistry("endpoint", endpoint));
        }

        public Endpoint(MetricRegistry registry) {
            this.registry = registry;

            started = registry.rate("staff.client.request.started");
            completed = registry.rate("staff.client.request.completed");
            inFlight = registry.lazyGaugeInt64("staff.client.request.inFlight",
                () -> started.get() - completed.get());

            elapsedTimeMillis = registry.histogramRate("staff.client.request.elapsedTimeMillis", Histograms.exponential(13, 2, 16));
        }

        public void started() {
            this.started.inc();
        }

        public void status(int statusCode) {
            statusCodes.computeIfAbsent(statusCode, code -> registry.rate("staff.client.request.status", Labels.of("code", code.toString()))).inc();
        }

        public void complete(long elapsedTimeNanos) {
            completed.inc();
            elapsedTimeMillis.record(TimeUnit.NANOSECONDS.toMillis(elapsedTimeNanos));
        }
    }
}
