package ru.yandex.yasms.http;

import java.io.IOException;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillNotClose;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.xml.XmlMapper;
import io.netty.handler.codec.http.HttpStatusClass;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.Request;
import org.asynchttpclient.RequestBuilder;

import ru.yandex.yasms.Status;
import ru.yandex.yasms.YasmsClient;
import ru.yandex.yasms.YasmsClientOptions;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class HttpYasmsClient implements YasmsClient {
    private final AsyncHttpClient client;

    private final YasmsClientOptions opts;
    private final ObjectMapper mapper;
    private final YasmsClientMetrics metrics;

    public HttpYasmsClient(@WillNotClose AsyncHttpClient client, YasmsClientOptions opts) {
        this.client = client;
        this.opts = opts;

        this.mapper = new XmlMapper();
        this.metrics = new YasmsClientMetrics(opts.getRegistry());
    }

    @Override
    public CompletableFuture<Status> sendToPhone(String phone, String text, String identity) {
        return exec(makePhoneRequest(phone, text, identity));
    }

    private CompletableFuture<Status> exec(Request request) {
        metrics.started();
        long startNanos = System.nanoTime();
        return client.executeRequest(request)
                .toCompletableFuture()
                .thenApply(response -> {
                    if (HttpStatusClass.SUCCESS.contains(response.getStatusCode())) {
                        return parseResponse(response.getResponseBody());
                    }

                    if (HttpStatusClass.CLIENT_ERROR.contains(response.getStatusCode())) {
                        return new Status(Status.Code.INVALID_REQUEST, response.getStatusText());
                    }

                    if (HttpStatusClass.SERVER_ERROR.contains(response.getStatusCode())) {
                        return new Status(Status.Code.INTERNAL_ERROR, response.getStatusText());
                    }

                    throw new UnsupportedOperationException("Unsupported status code: "
                            + response.getStatusCode()
                            + ": "
                            + response.getStatusText());
                })
                .whenComplete((status, e) -> {
                    long elapsedNanos = System.nanoTime() - startNanos;
                    if (e != null) {
                        metrics.complete(elapsedNanos, Status.Code.INTERNAL_ERROR);
                    } else {
                        metrics.complete(elapsedNanos, status.getCode());
                    }
                });
    }

    private Status parseResponse(String body) {
        SendSmsResponse response;
        try {
            response = mapper.readValue(body, SendSmsResponse.class);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        if (response.isSuccess()) {
            return new Status(Status.Code.SUCCESS, "Message-id: " + response.getMessageId());
        }

        Status.Code code = Status.Code.byCode(response.getErrorCode());
        return new Status(code, response.getError());
    }

    private Request makePhoneRequest(String phone, String text, String identity) {
        return new RequestBuilder("GET")
                .setUrl(opts.getAddress() + "/sendsms")
                .addQueryParam("sender", opts.getSender())
                .addQueryParam("text", text)
                .addQueryParam("phone", phone)
                .addQueryParam("utf8", "1")
                .addQueryParam("identity", identity)
                .build();
    }
}
