package ru.yandex.solomon.yasm.alert.converter;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Charsets;
import com.google.common.base.Joiner;

import ru.yandex.ambry.dto.YasmAlertDto;
import ru.yandex.solomon.alert.protobuf.NoPointsPolicy;
import ru.yandex.solomon.alert.protobuf.ResolvedEmptyPolicy;
import ru.yandex.solomon.alert.protobuf.TAlert;
import ru.yandex.solomon.alert.protobuf.TExpression;
import ru.yandex.solomon.util.collection.Nullables;
import ru.yandex.solomon.yasm.expression.ast.YasmAst;
import ru.yandex.solomon.yasm.expression.grammar.YasmExpression;
import ru.yandex.solomon.yasm.expression.grammar.YasmSelRenderer;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class YasmAlertConverter {

    public static final String NAME_PREFIX = "[YasmConverted] ";

    private final String yasmItypeProjectPrefix;

    public YasmAlertConverter(String yasmItypeProjectPrefix) {
        this.yasmItypeProjectPrefix = yasmItypeProjectPrefix;
    }

    public String projectFromItype(String itype) {
        return yasmItypeProjectPrefix + itype;
    }

    public static String alertIdFromYasmAlertName(String name) {
        // Yasm names does not follow solomon id restriction (neither length, nor special symbols)
        return UUID.nameUUIDFromBytes(("yasm_converted." + name).getBytes(Charsets.UTF_8)).toString();
    }

    public TAlert.Builder convertAlert(YasmAlertDto alert) {
        String signal = Nullables.orEmpty(alert.signal);
        String name = Nullables.orEmpty(alert.name);
        String program = translate(signal, alert.tags, alert);
        String itype = alert.tags.get("itype").get(0);

        int valueWindow = Optional.ofNullable(alert.valueModify)
                .flatMap(valueModify -> Optional.ofNullable(valueModify.window))
                .orElse(0);
        int intervalWindow = Optional.ofNullable(alert.interval)
                .orElse(0);

        int alertWindow = valueWindow + intervalWindow;
        if (alertWindow == 0) {
            alertWindow = 300;
        }

        return TAlert.newBuilder()
                .setId(alertIdFromYasmAlertName(name))
                .setName(NAME_PREFIX + name)
                .setProjectId(projectFromItype(itype))
                .setDescription(Nullables.orEmpty(alert.description))
                .setPeriodMillis(TimeUnit.SECONDS.toMillis(alertWindow))
                .setResolvedEmptyPolicy(ResolvedEmptyPolicy.RESOLVED_EMPTY_MANUAL)
                .setNoPointsPolicy(NoPointsPolicy.NO_POINTS_MANUAL)
                .setExpression(TExpression.newBuilder()
                        .setProgram(program));
    }

    @Nullable
    private static String formatCondition(@Nullable List<Double> bounds, String var) {
        if (bounds == null || bounds.size() != 2) {
            throw new IllegalArgumentException("Thresholds should be an array with two elements, had: " + bounds);
        }
        Double lower = bounds.get(0);
        Double upper = bounds.get(1);
        String lowerCondition = Nullables.map(lower, threshold -> var + " >= " + threshold);
        String upperCondition = Nullables.map(upper, threshold -> var + " <= " + threshold);
        if (lowerCondition == null && upperCondition == null) {
            return null;
        }
        if (lowerCondition != null && upperCondition != null) {
            return lowerCondition + " && " + upperCondition;
        }
        return Objects.requireNonNullElse(lowerCondition, upperCondition);
    }

    @Nullable
    private static String formatCondition(@Nullable Double bound, String var) {
        return Nullables.map(bound, value -> var + " >= " + bound);
    }

    private String translate(String signal, Map<String, List<String>> tags, YasmAlertDto alert) {
        StringBuilder builder = new StringBuilder();
        builder.append("use ");
        builder.append(YasmSelRenderer.renderTags(tags, yasmItypeProjectPrefix));
        builder.append(";\n");

        YasmAst ast = YasmExpression.parse(signal);
        YasmSelRenderer.RenderResult renderResult = YasmSelRenderer.render(ast, yasmItypeProjectPrefix);
        String signalExpression = renderResult.expression;
        for (var entry : renderResult.constants.entrySet()) {
            builder.append("let ").append(entry.getKey()).append(" = ").append(entry.getValue()).append(";\n");
        }

        builder.append("// ").append(signal).append('\n');

        if (alert.trend != null) {
            builder.append(translateTrendAlert("down".equals(alert.trend), signalExpression, alert));
        } else {
            builder.append(translateRegularAlert(signalExpression, alert));
        }
        return builder.toString();
    }

    private static String translateRegularAlert(String signalExpression, YasmAlertDto alert) {
        String data = "let signal = " + signalExpression + ";";
        String value = "\nlet value = " + formatAggregation(alert.valueModify) + "(signal);";
        String alarmCondition = formatCondition(alert.crit, "value");
        String warnCondition = formatCondition(alert.warn, "value");
        String noDataIf = "\nno_data_if(value != value);";
        String alarmIf = alarmCondition == null ? "" : "\nalarm_if(" + alarmCondition + ");";
        String warnIf = warnCondition == null ? "" : "\nwarn_if(" + warnCondition + ");";
        return data + value + noDataIf + warnIf + alarmIf;
    }

    private static String translateTrendAlert(boolean trendDown, String signalExpression, YasmAlertDto alert) {
        String data = "let signal = " + signalExpression + ";";
        String test;
        int valueWindow = (alert.valueModify == null || alert.valueModify.window == null)
                ? 0 : alert.valueModify.window;
        if  (valueWindow == 0) {
            test = "let test = signal;";
        } else {
            test = "let test = tail(signal, " + valueWindow + "s);";
        }
        String testValue = "let testValue = " + formatAggregation(alert.valueModify) + "(test);";
        int dropTrainWindow = (alert.intervalModify == null || alert.intervalModify.intervalEndOffset == null)
                ? valueWindow : alert.intervalModify.intervalEndOffset;

        String train;
        if (dropTrainWindow == 0) {
            train = "let train = signal;";
        } else {
            train = "let train = drop_tail(signal, " + dropTrainWindow + "s);";
        }
        String trainValue = "let trainValue = " + formatTrainAggregation(trendDown, alert.intervalModify) + ";";

        String alarmCondition, warnCondition, delta;
        if (alert.warnPerc != null || alert.critPerc != null) {
            // percentage based
            delta = "let deltaPerc = 100 * (" +
                        (trendDown ? "trainValue - testValue" : "testValue - trainValue") +
                    ") / trainValue;";
            alarmCondition = formatCondition(alert.critPerc, "deltaPerc");
            warnCondition = formatCondition(alert.warnPerc, "deltaPerc");
        } else {
            // absolute based
            delta = "let delta = " + (trendDown ? "trainValue - testValue" : "testValue - trainValue") + ";";
            alarmCondition = formatCondition(alert.crit, "delta");
            warnCondition = formatCondition(alert.warn, "delta");
        }

        String alarmIf = alarmCondition == null ? "" : "\nalarm_if(" + alarmCondition + ");";
        String warnIf = warnCondition == null ? "" : "\nwarn_if(" + warnCondition + ");";
        return Joiner.on("\n").join(
                data,
                train,
                test,
                trainValue,
                testValue,
                delta
        ) + alarmIf + warnIf;
    }

    private static final Map<String, String> TYPE_TO_AGGR = Map.of(
            "aver", "avg",
            "max", "max",
            "min", "min",
            "summ", "sum"
    );

    private static String formatAggregation(@Nullable YasmAlertDto.ValueModify valueModify) {
        if (valueModify == null || valueModify.type == null) {
            return "last";
        }
        String aggr = TYPE_TO_AGGR.get(valueModify.type);
        if (aggr == null) {
            throw new IllegalArgumentException("Type " + valueModify.type + " is not supported in valueModify");
        }
        return aggr;
    }

    private static String formatTrainAggregation(boolean trendDown, @Nullable YasmAlertDto.IntervalModify intervalModify) {
        if (intervalModify == null || intervalModify.type == null) {
            return trendDown ? "max(train)" : "min(train)";
        }
        if ("quant".equals(intervalModify.type)) {
            return "percentile(" + Nullables.orDefault(intervalModify.quant, 50) + ", train)";
        }
        String aggr = TYPE_TO_AGGR.get(intervalModify.type);
        if (aggr == null) {
            throw new IllegalArgumentException("Type " + intervalModify.type + " is not supported in intervalModify");
        }
        return aggr + "(train)";
    }
}
