package ru.yandex.solomon.yasm.expression.grammar;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;

import ru.yandex.solomon.yasm.expression.ast.YasmAst;
import ru.yandex.solomon.yasm.expression.ast.YasmAstIdent;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class SignalPattern {
    private final String formatPatterns;
    private final List<List<String>> alternatives;

    private static final char PAT_BEGIN = '<';
    private static final char PAT_SEP = '|';
    private static final char PAT_END = '>';

    private enum State {
        FIXED,
        PATTERN
    };

    public SignalPattern(String ident) {
        StringTokenizer tokenizer = new StringTokenizer(ident, "<|>", true);
        State state = State.FIXED;
        StringBuilder format = new StringBuilder(4);
        alternatives = new ArrayList<>(3);
        ArrayList<String> lastAlternatives = null;
        while (tokenizer.hasMoreTokens()) {
            String token = tokenizer.nextToken();
            char lead = token.charAt(0);
            switch (lead) {
                case PAT_BEGIN:
                    Preconditions.checkState(state == State.FIXED, "< inside a pattern");
                    state = State.PATTERN;
                    format.append("{").append(alternatives.size()).append("}");
                    lastAlternatives = new ArrayList<>();
                    alternatives.add(lastAlternatives);
                    break;
                case PAT_END:
                    Preconditions.checkState(state == State.PATTERN, "> outside a pattern");
                    state = State.FIXED;
                    break;
                case PAT_SEP:
                    Preconditions.checkState(state == State.PATTERN, "| outside a pattern");
                    break;
                default:
                    if (state == State.FIXED) {
                        format.append(token);
                    } else {
                        lastAlternatives.add(token);
                    }
            }
        }
        formatPatterns = format.toString();
        Preconditions.checkState(state == State.FIXED, "Unterminated pattern");
    }

    public Stream<YasmAst> generate() {
        if (alternatives.isEmpty()) {
            return Stream.of(new YasmAstIdent(formatPatterns));
        }
        return Lists.cartesianProduct(alternatives)
                .stream()
                .map(patternCase -> new YasmAstIdent(MessageFormat.format(formatPatterns, patternCase.toArray(new String[0]))));
    }
}
