package ru.yandex.solomon.yasm.expression.grammar;

import java.util.BitSet;

import javax.annotation.ParametersAreNonnullByDefault;

import org.antlr.v4.runtime.ANTLRErrorListener;
import org.antlr.v4.runtime.CharStreams;
import org.antlr.v4.runtime.CommonTokenStream;
import org.antlr.v4.runtime.DiagnosticErrorListener;
import org.antlr.v4.runtime.Parser;
import org.antlr.v4.runtime.RecognitionException;
import org.antlr.v4.runtime.Recognizer;
import org.antlr.v4.runtime.atn.ATNConfigSet;
import org.antlr.v4.runtime.dfa.DFA;
import org.antlr.v4.runtime.misc.Interval;

import ru.yandex.solomon.yasm.expression.ast.YasmAst;
import ru.yandex.solomon.yasm.expression.grammar.exceptions.AmbiguousParsingException;
import ru.yandex.solomon.yasm.expression.grammar.exceptions.AttemptingFullContextException;
import ru.yandex.solomon.yasm.expression.grammar.exceptions.ContextSensitivityException;
import ru.yandex.solomon.yasm.expression.grammar.exceptions.SyntaxErrorException;
import ru.yandex.solomon.yasm.grammar.generated.YasmLexer;
import ru.yandex.solomon.yasm.grammar.generated.YasmParser;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class YasmExpression {
    public static YasmAst parse(String code) {
        var lexer = new YasmLexer(CharStreams.fromString(code));
        var tokens = new CommonTokenStream(lexer);
        var parser = new YasmParser(tokens);

        lexer.removeErrorListeners();
        parser.removeErrorListeners();

        ANTLRErrorListener errorListener = new ThrowingErrorListener();
        lexer.addErrorListener(errorListener);
        parser.addErrorListener(errorListener);

        var astBuilder = new YasmAstVisitor();
        return astBuilder.visitInput(parser.input());
    }

    private static class ThrowingErrorListener extends DiagnosticErrorListener {
        @Override
        public void syntaxError(
                Recognizer<?, ?> recognizer,
                Object offendingSymbol,
                int line,
                int charPositionInLine,
                String msg,
                RecognitionException e)
        {
            throw new SyntaxErrorException("line " + line + ":" + charPositionInLine + " " + msg, e);
        }

        @Override
        public void reportAmbiguity(
                Parser recognizer,
                DFA dfa,
                int startIndex,
                int stopIndex,
                boolean exact,
                BitSet ambigAlts,
                ATNConfigSet configs)
        {
            String format = "reportAmbiguity d=%s: ambigAlts=%s, input='%s'";
            String decision = getDecisionDescription(recognizer, dfa);
            BitSet conflictingAlts = getConflictingAlts(ambigAlts, configs);
            String text = recognizer.getTokenStream().getText(Interval.of(startIndex, stopIndex));
            String message = String.format(format, decision, conflictingAlts, text);
            throw new AmbiguousParsingException(message);
        }

        @Override
        public void reportAttemptingFullContext(
                Parser recognizer,
                DFA dfa,
                int startIndex,
                int stopIndex,
                BitSet conflictingAlts,
                ATNConfigSet configs)
        {
            String format = "reportAttemptingFullContext d=%s, input='%s'";
            String decision = getDecisionDescription(recognizer, dfa);
            String text = recognizer.getTokenStream().getText(Interval.of(startIndex, stopIndex));
            String message = String.format(format, decision, text);
            throw new AttemptingFullContextException(message);
        }

        @Override
        public void reportContextSensitivity(
                Parser recognizer,
                DFA dfa,
                int startIndex,
                int stopIndex,
                int prediction,
                ATNConfigSet configs)
        {
            String format = "reportContextSensitivity d=%s, input='%s'";
            String decision = getDecisionDescription(recognizer, dfa);
            String text = recognizer.getTokenStream().getText(Interval.of(startIndex, stopIndex));
            String message = String.format(format, decision, text);
            throw new ContextSensitivityException(message);
        }
    }
}
