package ru.yandex.solomon.yasm.expression.grammar;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.expression.PositionRange;
import ru.yandex.solomon.expression.ast.Ast;
import ru.yandex.solomon.expression.ast.AstCall;
import ru.yandex.solomon.expression.ast.AstIdent;
import ru.yandex.solomon.expression.ast.AstSelector;
import ru.yandex.solomon.expression.ast.AstSelectors;
import ru.yandex.solomon.expression.ast.AstValueDouble;
import ru.yandex.solomon.expression.ast.AstValueString;
import ru.yandex.solomon.expression.ast.serialization.AstMappingContext;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.SelectorType;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.SelectorsBuilder;
import ru.yandex.solomon.yasm.expression.ast.YasmAst;
import ru.yandex.solomon.yasm.expression.ast.YasmAstCall;
import ru.yandex.solomon.yasm.expression.ast.YasmAstIdent;
import ru.yandex.solomon.yasm.expression.ast.YasmAstNumber;
import ru.yandex.solomon.yasm.expression.grammar.functions.Aver;
import ru.yandex.solomon.yasm.expression.grammar.functions.Const;
import ru.yandex.solomon.yasm.expression.grammar.functions.Conv;
import ru.yandex.solomon.yasm.expression.grammar.functions.Diff;
import ru.yandex.solomon.yasm.expression.grammar.functions.Div;
import ru.yandex.solomon.yasm.expression.grammar.functions.Havg;
import ru.yandex.solomon.yasm.expression.grammar.functions.HcountHperc;
import ru.yandex.solomon.yasm.expression.grammar.functions.Hmerge;
import ru.yandex.solomon.yasm.expression.grammar.functions.Hsum;
import ru.yandex.solomon.yasm.expression.grammar.functions.MinMax;
import ru.yandex.solomon.yasm.expression.grammar.functions.Mul;
import ru.yandex.solomon.yasm.expression.grammar.functions.Neg;
import ru.yandex.solomon.yasm.expression.grammar.functions.Normal;
import ru.yandex.solomon.yasm.expression.grammar.functions.Or;
import ru.yandex.solomon.yasm.expression.grammar.functions.Perc;
import ru.yandex.solomon.yasm.expression.grammar.functions.Quant;
import ru.yandex.solomon.yasm.expression.grammar.functions.Sum;
import ru.yandex.solomon.yasm.expression.grammar.functions.UnaryFunc;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class YasmSelRenderer implements YasmAstElementVisitor<ExpressionWithConstants> {
    private static final Logger logger = LoggerFactory.getLogger(YasmSelRenderer.class);

    private static final String SIGNAL = "signal";
    private static final String ITYPE = "itype";
    private static final Map<String, FunctionRenderer> SUPPORTED_CALLS;
    private static final PositionRange U = PositionRange.UNKNOWN;

    private static final Pattern IS_SIGNAL_SUFFIX_VALID = Pattern.compile("([advehmntx]{4}|summ|hgram|max)");

    private final String yasmProjectPrefix;
    private final Map<String, List<String>> tags;

    static {
        SUPPORTED_CALLS = new HashMap<>();

        registerRenderer(new Perc());
        registerRenderer(new Div());
        registerRenderer(new Diff());
        registerRenderer(new Mul());
        registerRenderer(new Sum());

        for (String name : List.of("ceil", "floor", "round", "trunc", "fract", "log", "exp", "sqrt", "abs")) {
            registerRenderer(new UnaryFunc(name));
        }
        registerRenderer(new Neg());

        registerRenderer(new Conv());
        registerRenderer(new Const());
        registerRenderer(new MinMax("min"));
        registerRenderer(new MinMax("max"));
        registerRenderer(new Aver());

        registerRenderer(new Quant());
        registerRenderer(new Hsum());
        registerRenderer(new Havg());
        registerRenderer(new Hmerge());
        registerRenderer(new HcountHperc("hcount", "histogram_count"));
        registerRenderer(new HcountHperc("hperc", "histogram_cdfp"));

        registerRenderer(new Or());
        registerRenderer(new Normal());
    }

    private static void registerRenderer(FunctionRenderer renderer) {
        SUPPORTED_CALLS.put(renderer.name(), renderer);
    }

    public YasmSelRenderer(String yasmProjectPrefix, Map<String, List<String>> tags) {
        this.yasmProjectPrefix = yasmProjectPrefix;
        this.tags = tags;
    }

    private YasmSelRenderer(String yasmProjectPrefix) {
        this.yasmProjectPrefix = yasmProjectPrefix;
        this.tags = Map.of();
    }

    public static class RenderResult {
        public String expression;
        public Map<String, SelValue> constants;

        RenderResult(ExpressionWithConstants visit, AstMappingContext ctx) {
            this.constants = visit.constants;
            this.expression = ctx.renderToString(visit.expression);
        }
    }

    public static RenderResult render(YasmAst ast, String yasmProjectPrefix) {
        AstMappingContext mappingContext = new AstMappingContext(false);
        return new RenderResult(new YasmSelRenderer(yasmProjectPrefix).visit(ast), mappingContext);
    }

    public static RenderResult render(YasmAst ast, Map<String, List<String>> tags, String yasmProjectPrefix) {
        AstMappingContext mappingContext = new AstMappingContext(false);
        return new RenderResult(new YasmSelRenderer(yasmProjectPrefix, tags).visit(ast), mappingContext);
    }

    public static String renderTags(Map<String, List<String>> tags, String yasmProjectPrefix) {
        AstMappingContext mappingContext = new AstMappingContext(false);
        return mappingContext.renderToString(new YasmSelRenderer(yasmProjectPrefix).formatUsingFromTags(tags));
    }

    private Ast formatUsingFromTags(Map<String, List<String>> tags) {
        return addOneMoreWrapToSelectors(tagsToSelectors(tags, null));
    }

    private List<AstSelector> formatSelectors(Selectors selectors) {
        return selectors.stream().map(selector ->
                        new AstSelector(U,
                                new AstValueString(U, selector.getKey(), true),
                                new AstValueString(U, selector.getValue()),
                                selector.getType()))
                .collect(Collectors.toList());
    }

    private Selectors tagsToSelectors(Map<String, List<String>> tags, @Nullable String signal) {
        var builder = new SelectorsBuilder("", 2 + tags.size());
        List<String> itypes = tags.get(ITYPE);
        if (itypes == null) {
            throw new IllegalArgumentException("itype is missing in tags");
        }
        if (itypes.size() != 1) {
            throw new IllegalArgumentException("itype tag must have single value");
        }
        String itype = itypes.get(0);
        builder.add(LabelKeys.PROJECT, yasmProjectPrefix + itype);
        for (var entry : tags.entrySet()) {
            if (entry.getKey().equals(ITYPE)) {
                continue;
            }
            builder.add(entry.getKey(), String.join("|", entry.getValue()));
        }
        if (signal != null) {
            builder.add(SIGNAL, signal);
        }
        return builder.build();
    }

    @Override
    public ExpressionWithConstants visitCall(YasmAstCall call) {
        var handler = SUPPORTED_CALLS.get(call.getFunc());
        if (handler == null) {
            throw new UnsupportedOperationException("Function " + call.getFunc() + " is not supported yet");
        }
        return handler.render(this, call.getArgs());
    }

    @Override
    public ExpressionWithConstants visitNumber(YasmAstNumber number) {
        return ExpressionWithConstants.scalar(
                new AstValueDouble(U, number.getValue())
        );
    }

    @Override
    public ExpressionWithConstants visitIdent(YasmAstIdent signal) {
        if (!tags.isEmpty()) {
            return ExpressionWithConstants.series(
                    makeSignalSelectorWithTags(signal.getIdent(), tags)
            );
        }
        return ExpressionWithConstants.series(
                makeSignalSelectors(signal.getIdent())
        );
    }

    private Ast makeSignalSelectorWithTags(String signal, Map<String, List<String>> tags) {
        return addOneMoreWrapToSelectors(tagsToSelectors(tags, signal));
    }

    private Ast addOneMoreWrapToSelectors(Selectors selectors){
        if (isOneMoreWrapNeeded(selectors)) {
            String signal = selectors.findByKey(SIGNAL).getValue();
            String suffix = signal.substring(signal.lastIndexOf('_') + 1);
            if (IS_SIGNAL_SUFFIX_VALID.matcher(suffix).find()) {
                String ident = switch (suffix.charAt(2)) {
                    case 'r', 'h', 'm', 'e' -> "series_sum";
                    case 'n' -> "series_min";
                    case 'x' -> "series_max";
                    case 'v' -> "series_avg";
                    default -> "";
                };
                if (!ident.isEmpty()) {
                    return new AstCall(U, new AstIdent(U, ident),
                            List.of(new AstValueString(U,"signal"),
                                    new AstSelectors(U, "", formatSelectors(selectors))));
                }
            }
        }
        return new AstSelectors(U, "", formatSelectors(selectors));
    }

    private boolean isOneMoreWrapNeeded(Selectors selectors) {
        for (Selector selector : selectors) {
            if (!selector.getKey().equals("hosts") &&
                    !selector.getKey().equals(SIGNAL) &&
                    !selector.isExact()) {
                return selectors.hasKey(SIGNAL);
            }
        }
        return false;
    }

    private static Ast makeSignalSelectors(String signal) {
        return new AstSelectors(U, "", List.of(
                new AstSelector(U,
                        new AstValueString(U, SIGNAL, true),
                        new AstValueString(U, signal),
                        SelectorType.GLOB))
        );
    }
}
