package ru.yandex.solomon.yasm.expression.grammar.functions;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.PositionRange;
import ru.yandex.solomon.expression.ast.AstBinOp;
import ru.yandex.solomon.expression.ast.AstIdent;
import ru.yandex.solomon.expression.ast.AstOp;
import ru.yandex.solomon.expression.ast.AstValueDouble;
import ru.yandex.solomon.expression.value.SelValueDouble;
import ru.yandex.solomon.yasm.expression.ast.YasmAst;
import ru.yandex.solomon.yasm.expression.ast.YasmAstIdent;
import ru.yandex.solomon.yasm.expression.ast.YasmAstNumber;
import ru.yandex.solomon.yasm.expression.grammar.ExpressionWithConstants;
import ru.yandex.solomon.yasm.expression.grammar.FunctionRenderer;
import ru.yandex.solomon.yasm.expression.grammar.YasmSelRenderer;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class Conv implements FunctionRenderer {

    private static final Map<String, Double> SUPPORTED_CONSTANTS;

    static {
        SUPPORTED_CONSTANTS = new HashMap<>();
        SUPPORTED_CONSTANTS.put("Ki", 1024d);
        SUPPORTED_CONSTANTS.put("Mi", Math.pow(1024d, 2));
        SUPPORTED_CONSTANTS.put("Gi", Math.pow(1024d, 3));
        SUPPORTED_CONSTANTS.put("Ti", Math.pow(1024d, 4));
        SUPPORTED_CONSTANTS.put("Pi", Math.pow(1024d, 5));

        SUPPORTED_CONSTANTS.put("u", 1e-6);
        SUPPORTED_CONSTANTS.put("m", 1e-3);
        SUPPORTED_CONSTANTS.put("c", 1e-2);
        SUPPORTED_CONSTANTS.put("d", 1e-1);

        SUPPORTED_CONSTANTS.put("k", 1e3);
        SUPPORTED_CONSTANTS.put("M", 1e6);
        SUPPORTED_CONSTANTS.put("G", 1e9);
        SUPPORTED_CONSTANTS.put("T", 1e12);
    }

    static ExpressionWithConstants renderUnit(YasmAst unit) {
        if (unit instanceof YasmAstIdent) {
            String ident = ((YasmAstIdent) unit).getIdent();
            Double value = SUPPORTED_CONSTANTS.get(ident);
            if (value == null) {
                throw new IllegalArgumentException("Unknown constant " + ident);
            }
            return ExpressionWithConstants.scalar(
                    new AstIdent(PositionRange.UNKNOWN, ident),
                    Map.of(ident, new SelValueDouble(value))
            );
        }
        if (unit instanceof YasmAstNumber) {
            double value = ((YasmAstNumber) unit).getValue();
            return ExpressionWithConstants.scalar(new AstValueDouble(PositionRange.UNKNOWN, value));
        }
        throw new IllegalArgumentException("Bad ast type for unit: " + unit.getType());
    }

    @Override
    public String name() {
        return "conv";
    }

    @Override
    public ExpressionWithConstants render(YasmSelRenderer renderer, List<YasmAst> args) {
        if (args.size() == 2) {
            // conv(signal, unitTo)
            ExpressionWithConstants signal = renderer.visit(args.get(0));
            ExpressionWithConstants unitTo = renderUnit(args.get(1));
            return ExpressionWithConstants.series(
                    new AstBinOp(
                            PositionRange.UNKNOWN,
                            signal.expression,
                            unitTo.expression,
                            new AstOp(PositionRange.UNKNOWN, "/")
                    ),
                    List.of(signal.constants, unitTo.constants)
            );
        }
        if (args.size() == 3) {
            // conv(signal, unitFrom, unitTo)
            ExpressionWithConstants signal = renderer.visit(args.get(0));
            ExpressionWithConstants unitFrom = renderUnit(args.get(1));
            ExpressionWithConstants unitTo = renderUnit(args.get(2));
            return ExpressionWithConstants.series(
                    new AstBinOp(
                            PositionRange.UNKNOWN,
                            signal.expression,
                            new AstBinOp(
                                    PositionRange.UNKNOWN,
                                    unitFrom.expression,
                                    unitTo.expression,
                                    new AstOp(PositionRange.UNKNOWN, "/")
                            ),
                            new AstOp(PositionRange.UNKNOWN, "*")
                    ),
                    List.of(signal.constants, unitFrom.constants, unitTo.constants)
            );
        }
        throw new IllegalArgumentException("Expected 2 or 3 arguments for conv call, got: " + args.size());
    }
}
