package ru.yandex.solomon.yasm.expression.grammar.functions;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.expression.PositionRange;
import ru.yandex.solomon.expression.ast.Ast;
import ru.yandex.solomon.expression.ast.AstCall;
import ru.yandex.solomon.expression.ast.AstIdent;
import ru.yandex.solomon.expression.ast.AstOp;
import ru.yandex.solomon.expression.ast.AstUnaryOp;
import ru.yandex.solomon.expression.ast.AstValueDouble;
import ru.yandex.solomon.expression.ast.AstValueString;
import ru.yandex.solomon.yasm.expression.ast.YasmAst;
import ru.yandex.solomon.yasm.expression.ast.YasmAstIdent;
import ru.yandex.solomon.yasm.expression.ast.YasmAstNumber;
import ru.yandex.solomon.yasm.expression.grammar.ExpressionWithConstants;
import ru.yandex.solomon.yasm.expression.grammar.FunctionRenderer;
import ru.yandex.solomon.yasm.expression.grammar.YasmSelRenderer;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class HcountHperc implements FunctionRenderer {
    private final String name;
    private final String selName;

    private static final AstCall INF = new AstCall(PositionRange.UNKNOWN,
            new AstIdent(PositionRange.UNKNOWN, "inf"),
            List.of());
    private static final AstUnaryOp NEG_INF = new AstUnaryOp(PositionRange.UNKNOWN,
            INF,
            new AstOp(PositionRange.UNKNOWN, "-"));

    public HcountHperc(String name, String selName) {
        this.name = name;
        this.selName = selName;
    }

    @Override
    public String name() {
        return name;
    }

    @Override
    public ExpressionWithConstants render(YasmSelRenderer renderer, List<YasmAst> args) {
        if (args.size() < 1 || args.size() > 3) {
            throw new IllegalArgumentException("Expected from 1 to 3 arguments for " + name + " call, got: " + args.size());
        }
        Ast lowerBound = args.size() >= 2 ? parseLowerBound(args.get(1)) : NEG_INF;
        Ast upperBound = args.size() == 3 ? parseUpperBound(args.get(2)) : INF;
        List<Ast> selArgs;
        ExpressionWithConstants signal = renderer.visit(args.get(0));
        AstValueString empty = new AstValueString(PositionRange.UNKNOWN, "");
        if (lowerBound != NEG_INF) {
            selArgs = List.of(lowerBound, upperBound, empty, signal.expression);
        } else if (upperBound != INF) {
            selArgs = List.of(upperBound, empty, signal.expression);
        } else {
            selArgs = List.of(empty, signal.expression);
        }
        return ExpressionWithConstants.series(new AstCall(PositionRange.UNKNOWN,
                new AstIdent(PositionRange.UNKNOWN, selName),
                selArgs), signal.constants);
    }

    private Ast parseLowerBound(YasmAst ast) {
        if (ast instanceof YasmAstIdent) {
            String ident = ((YasmAstIdent) ast).getIdent();
            if (ident.equals("inf") || ident.equals("-inf")) {
                return NEG_INF;
            } else {
                throw new IllegalArgumentException("Bad lower bound in " + name + ", got: " + ident);
            }
        }
        if (ast instanceof YasmAstNumber) {
            double value = ((YasmAstNumber) ast).getValue();
            return new AstValueDouble(PositionRange.UNKNOWN, value);
        }
        throw new IllegalArgumentException("Bad argument for lower bound in " + name + ", got: " + ast.getType());
    }

    private Ast parseUpperBound(YasmAst ast) {
        if (ast instanceof YasmAstIdent) {
            String ident = ((YasmAstIdent) ast).getIdent();
            if (ident.equals("inf")) {
                return INF;
            } else {
                throw new IllegalArgumentException("Bad upper bound in " + name + ", got: " + ident);
            }
        }
        if (ast instanceof YasmAstNumber) {
            double value = ((YasmAstNumber) ast).getValue();
            return new AstValueDouble(PositionRange.UNKNOWN, value);
        }
        throw new IllegalArgumentException("Bad argument for upper bound in " + name + ", got: " + ast.getType());
    }
}
