package ru.yandex.solomon.kikimr;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.function.Consumer;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.io.ByteStreams;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.devtools.test.Paths;


/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class Installer {

    private static final Logger logger = LoggerFactory.getLogger(Installer.class);

    public static final String KIKIMR_BINARY = "kikimr";
    public static final String LOCAL_YDB_BINARY = "local_ydb";
    public static final String UDFS_DIR = "udfs";

    public static void installIfNecessary(Path installDir, KikimrVersion version) throws IOException {
        Files.createDirectories(installDir);

        // Check if already unpacked - cache local runs from IDE
        Path installedBin = installDir.resolve(LOCAL_YDB_BINARY);
        if (Files.exists(installedBin)) {
            logger.info("found {} downloaded recently: {}", LOCAL_YDB_BINARY, installedBin);
            return;
        }

        // Look for resources downloaded via solomon/libs/java/ydb.inc in ya.make
        // Used in CI tests where network is restricted
        if (getKikimrDownloadedByYaMake(installDir, version)) {
            return;
        }

        logger.info("download kikimr from sandbox");
        downloadAndInstallFromSandbox(installDir, version);
    }

    private static boolean getKikimrDownloadedByYaMake(Path installDir, KikimrVersion version) throws IOException {
        @Nullable String sandboxRootStr = Paths.getSandboxResourcesRoot();
        if (sandboxRootStr == null) { // Not under ya make -t
            return false;
        }
        File sandboxRoot = Path.of(sandboxRootStr).toFile();

        logger.info("Looking for local ydb in " + sandboxRootStr);

        File[] matched = sandboxRoot.listFiles((dir, name) -> name.startsWith("yandex-solomon-local-ydb") &&
                                name.contains(version.getTagName()));

        if (matched == null || matched.length != 1) {
            logger.warn("Incorrect number of matching files");
            return false;
        }

        File archive = matched[0];
        logger.info("found {}", archive.getName());

        installFromStream(installDir, new FileInputStream(archive));
        return true;
    }

    private static void downloadAndInstallFromSandbox(Path installDir, KikimrVersion version) throws IOException {
        Long resourceId = version.getSandboxResourceIdFor(KikimrPlatform.CURRENT);
        Verify.verifyNotNull(resourceId, "Unsupported platform " + KikimrPlatform.CURRENT +
                " for version " + version);
        installFromStream(installDir, getSandboxResource(resourceId));
    }

    private static InputStream getSandboxResource(long id) {
        try {
            URL url = new URL("https://proxy.sandbox.yandex-team.ru/" + id);
            return url.openStream();
        } catch (MalformedURLException e) {
            throw new RuntimeException(e);
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private static void installFromStream(Path installDir, InputStream stream) throws IOException {
        Path localYdbPath = installDir.resolve(LOCAL_YDB_BINARY);
        if (Files.exists(localYdbPath)) {
            logger.info("Driver {} exists - skip installation", localYdbPath);
            return;
        }

        doInstall(installDir, stream, path -> Preconditions.checkState(Files.exists(path.resolve(LOCAL_YDB_BINARY))));
        logger.info("Kikimr driver has been installed to {}", localYdbPath);
    }

    public static void doInstall(Path installDir, InputStream archiveInput, Consumer<Path> verifier) throws IOException {
        Path tempArchiveFilePath = Files.createTempFile("local.kikimr.", ".tar.gz");
        try {
            logger.info("Copying archive into file {}", tempArchiveFilePath);
            try (FileOutputStream archiveOutput = new FileOutputStream(tempArchiveFilePath.toFile())) {
                ByteStreams.copy(archiveInput, archiveOutput);
            }
            Path tempDir = Files.createTempDirectory(installDir.getFileName().toString());

            Utils.extractTarGz(tempArchiveFilePath, tempDir);

            verifier.accept(tempDir);
            FileUtils.deleteDirectory(installDir.toFile());
            FileUtils.moveDirectory(tempDir.toFile(), installDir.toFile());

            Utils.makeExecutable(installDir.resolve(KIKIMR_BINARY));
            Utils.makeExecutable(installDir.resolve(LOCAL_YDB_BINARY));
        } finally {
            Files.deleteIfExists(tempArchiveFilePath);
        }
    }
}
