package ru.yandex.solomon.kikimr;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Optional;
import java.util.UUID;

import javax.annotation.Nullable;

import com.google.common.net.HostAndPort;
import org.apache.commons.io.FileUtils;
import org.junit.rules.ExternalResource;
import sun.misc.Signal;

import ru.yandex.devtools.test.Runner;

import static ru.yandex.solomon.kikimr.Installer.KIKIMR_BINARY;

/**
 * @author Vladimir Gordiychuk
 */
public class LocalKikimr extends ExternalResource {

    private static final Path INSTALL_DIR = Optional.ofNullable(Runner.params)
        .map(parameters -> Paths.get(parameters.buildRoot).getParent().resolve("local-kikimr"))
        .orElse(FileUtils.getUserDirectory().toPath().resolve(".kikimr"));

    private final KikimrVersion version;
    private final Path installDir;

    private LocalYdbManager manager;
    private HostAndPort grpcEndpoint;

    public LocalKikimr() {
        this(LocalKikimr.newBuilder());
    }

    private LocalKikimr(Builder builder) {
        this.version = builder.version;
        this.installDir = builder.installDir != null
            ? builder.installDir
            : INSTALL_DIR.resolve(version.name());
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    private void sigintHandler(Signal sig) {
        try {
            manager.stop();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    @Override
    protected void before() throws IOException {
        Installer.installIfNecessary(installDir, version);
        Path workDir = installDir.getParent().resolve("deploy-" + UUID.randomUUID());
        manager = new LocalYdbManager(installDir.resolve(KIKIMR_BINARY), workDir);
        Signal.handle(new Signal("INT"), this::sigintHandler);
        manager.start();
        grpcEndpoint = readGrpcEndpoint(workDir);
    }

    private HostAndPort readGrpcEndpoint(Path workDir) {
        try (BufferedReader reader = new BufferedReader(new FileReader(workDir.resolve("ydb_endpoint.txt").toFile()))) {
            return HostAndPort.fromString(reader.readLine());
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @Override
    protected void after() {
        try {
            manager.stop();
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public Path getRoot() {
        return Path.of("/local");
    }

    public HostAndPort getGrpcEndpoint() {
        if (grpcEndpoint == null) {
            throw new IllegalStateException("The server has not been started yet");
        }
        return grpcEndpoint;
    }

    public static class Builder {
        private KikimrVersion version = KikimrVersion.LATEST;

        @Nullable
        private Path installDir;

        private Builder() {
        }

        public Builder setVersion(KikimrVersion version) {
            this.version = version;
            return this;
        }

        public Builder setInstallDir(Path installDir) {
            this.installDir = installDir;
            return this;
        }

        public LocalKikimr build() {
            return new LocalKikimr(this);
        }
    }
}
