package ru.yandex.solomon.kikimr;

import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Verify;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class LocalYdbManager {
    private static final Logger logger = LoggerFactory.getLogger(LocalYdbManager.class);

    private final Path workingDirectory;
    private final Path kikimrDriver;
    private final Path localYdb;

    public LocalYdbManager(Path kikimrDriver, Path workingDirectory) {
        this.kikimrDriver = kikimrDriver;
        this.workingDirectory = workingDirectory;
        this.localYdb = kikimrDriver.getParent().resolve(Installer.LOCAL_YDB_BINARY);

        logger.debug("Creating working directory {}", workingDirectory);
        try {
            Verify.verify(Files.notExists(workingDirectory));
            Files.createDirectories(workingDirectory);
        } catch (IOException e) {
            logger.error("Creating working directory failed", e);
            throw new UncheckedIOException(e);
        }
    }

    private enum Action {
        DEPLOY,
        START,
        STOP,
        CLEANUP,
        ;
    }

    private void perform(Action action, String... args) {
        try {
            List<String> command = new ArrayList<>(List.of(
                    localYdb.toString(), action.toString().toLowerCase(Locale.ROOT),
                    "--ydb-working-dir", workingDirectory.toString(),
                    "--ydb-binary-path", kikimrDriver.toString(),
                    "--ydb-udfs-dir", kikimrDriver.getParent().resolve(Installer.UDFS_DIR).toString()));
            command.addAll(Arrays.stream(args).collect(Collectors.toList()));
            Utils.exec(command, workingDirectory.toFile());
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public void start() {
        perform(Action.DEPLOY);
    }

    public void stop() throws IOException {
        try {
            perform(Action.CLEANUP);
        } finally {
            if (workingDirectory.toFile().exists()) {
                logger.debug("Deleting working directory {}", workingDirectory);
                File dir = workingDirectory.toFile();
                if (!FileUtils.deleteQuietly(dir)) {
                    logger.debug("Could not delete server deployment dir {}. Try to delete on exit.", workingDirectory);
                    dir.deleteOnExit();
                }
            }
        }
    }
}
