package ru.yandex.solomon.kikimr;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.PosixFilePermission;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Joiner;
import com.google.common.base.Verify;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static java.lang.ProcessBuilder.Redirect.INHERIT;


/**
 * Utility class for the logic implementing local Kikimr server functionality.
 *
 * @author snoop
 */
@ParametersAreNonnullByDefault
public class Utils {

    private static final Logger logger = LoggerFactory.getLogger(Utils.class);

    private static final int OK = 0;
    private static final int DEFAULT_TRANSFER_SIZE = 30 << 20; // 30 MiB

    private Utils() {
    }

    public static void makeExecutable(Path path) throws IOException {
        Set<PosixFilePermission> perms = Files.getPosixFilePermissions(path);
        perms.add(PosixFilePermission.OWNER_EXECUTE);
        perms.add(PosixFilePermission.GROUP_EXECUTE);
        perms.add(PosixFilePermission.OTHERS_EXECUTE);
        Files.setPosixFilePermissions(path, perms);
    }

    public static void extractTarGz(Path archivePath, Path dir) throws IOException {
        try (FileInputStream gzipStream = new FileInputStream(archivePath.toFile());
             GzipCompressorInputStream tarStream = new GzipCompressorInputStream(gzipStream);
             TarArchiveInputStream archive = new TarArchiveInputStream(tarStream);
             ReadableByteChannel archiveChannel = Channels.newChannel(archive)
             )
        {
            while (true) {
                @Nullable TarArchiveEntry entry = archive.getNextTarEntry();
                if (entry == null) {
                    break;
                }

                if (entry.isFile()) {
                    readPartToFile(archiveChannel, entry.getSize(), dir.resolve(entry.getName()).toFile());
                }
            }
        }
    }

    private static void readPartToFile(ReadableByteChannel input, long size, File file) throws IOException {
        var ignored = file.getParentFile().mkdirs();

        try (FileOutputStream os = new FileOutputStream(file)) {
            FileChannel output = os.getChannel();

            long pos = 0L;
            while (pos < size) {
                long count = Math.min(size - pos, DEFAULT_TRANSFER_SIZE);
                pos += output.transferFrom(input, pos, count);
            }
        }
    }

    public static void exec(List<String> command, File workDir) throws IOException {
        long startMillis = System.currentTimeMillis();
        String execId = UUID.randomUUID().toString();
        logger.info("{} - exec {} using work dir {}", execId, Joiner.on(" ").join(command), workDir.toString());
        ProcessBuilder pb = new ProcessBuilder(command)
                .directory(workDir)
                .redirectOutput(ProcessBuilder.Redirect.INHERIT);
        var env = pb.environment();
        env.put("YDB_YQL_SYNTAX_VERSION", "1");
        pb.redirectError(INHERIT);
        Process process = pb.start();
        waitForSuccess(process, execId);
        long elapsedMillis = System.currentTimeMillis() - startMillis;
        logger.info("{} - command succeeded, took {} secs", execId, 1d * elapsedMillis / TimeUnit.SECONDS.toMillis(1));
    }

    private static void waitForSuccess(Process process, String cmdId) {
        try {
            process.waitFor();
            Verify.verify(process.exitValue() == OK,
                    "Command with id=%s exit with status %s", cmdId, process.exitValue());
        } catch (InterruptedException e) {
            throw new RuntimeException("Interruption could not allow to complete execution", e);
        }
    }
}
