package ru.yandex.solomon.ydb;

import java.time.Clock;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import com.google.common.annotations.VisibleForTesting;
import com.yandex.ydb.table.query.Params;
import io.grpc.Status;

import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;

import static com.google.common.base.Preconditions.checkState;


/**
 * @author Sergey Polovko
 */
public class AsyncFinder<Id extends Comparable<Id>, Entity> {

    private static final int MIN_PAGE_SIZE = 300;

    private final YdbTable<Id, Entity> table;
    private final Params params;
    private final Function<PageOptions, String> queryBuilder;
    private final CompletableFuture<PagedResult<Entity>> promise;
    private final Clock clock;

    @VisibleForTesting
    AsyncFinder(YdbTable<Id, Entity> table, Params params, Function<PageOptions, String> queryBuilder, Clock clock) {
        this.table = table;
        this.params = params;
        this.queryBuilder = queryBuilder;
        this.promise = new CompletableFuture<>();
        this.clock = clock;
    }

    AsyncFinder(YdbTable<Id, Entity> table, Params params, Function<PageOptions, String> queryBuilder) {
        this(table, params, queryBuilder, Clock.systemUTC());
    }

    CompletableFuture<PagedResult<Entity>> getFuture() {
        return promise;
    }

    // TODO: rewrite with proper paging, using max key from prev response as starting point of the next page.
    //       it is required to rewrite all paging requests
    void findAll(Instant deadline) {
        long deadlineMillis = deadline.toEpochMilli();
        findPage(params, PageOptions.ALL)
            .whenComplete((page, throwable) -> {
                if (throwable != null) {
                    promise.completeExceptionally(throwable);
                    return;
                }

                if (page.getTotalCount() <= page.getResult().size()) {
                    promise.complete(page);
                    return;
                }

                try {
                    TreeMap<Id, Entity> sink = new TreeMap<>();
                    for (Entity entity : page.getResult()) {
                        Id id = table.getId(entity);
                        checkState(sink.put(id, entity) == null, "duplicate id: %s", id);
                    }
                    PageOptions nextPageOpts = nextPageOpts(page, 1);
                    continueFind(nextPageOpts, sink, deadlineMillis);
                } catch (Throwable t) {
                    promise.completeExceptionally(t);
                }
            });
    }

    private void continueFind(PageOptions pageOpts, Map<Id, Entity> sink, long deadlineMillis) {
        if (clock.millis() > deadlineMillis) {
            promise.completeExceptionally(Status.DEADLINE_EXCEEDED
                    .withDescription("findAll timeout for table " + table.getPath() + " with params " + params.toPb())
                    .asRuntimeException());
            return;
        }

        findPage(params, pageOpts)
            .whenComplete((page, throwable) -> {
                if (throwable != null) {
                    promise.completeExceptionally(throwable);
                    return;
                }

                try {
                    if (page.getResult().isEmpty()) {
                        ArrayList<Entity> entities = new ArrayList<>(sink.values());
                        promise.complete(PagedResult.of(entities, PageOptions.ALL, entities.size()));
                        return;
                    }

                    for (Entity entity : page.getResult()) {
                        Id id = table.getId(entity);
                        // it is OK if we got overlapped pages here, just keep last seen entity
                        sink.put(id, entity);
                    }

                    PageOptions nextPageOpts = nextPageOpts(page, pageOpts.getCurrent() + 1);
                    continueFind(nextPageOpts, sink, deadlineMillis);
                } catch (Throwable t) {
                    promise.completeExceptionally(t);
                }
            });
    }

    private CompletableFuture<PagedResult<Entity>> findPage(Params params, PageOptions pageOpts) {
        try {
            String query = queryBuilder.apply(pageOpts);
            return table.queryPage(query, params, pageOpts);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    private static PageOptions nextPageOpts(PagedResult<?> page, int pageNum) {
        // limit minimal page size to avoid a lot of queries with page size == 1
        int nextPageSize = Math.max(MIN_PAGE_SIZE, page.getResult().size());
        return new PageOptions(nextPageSize, pageNum);
    }
}
