package ru.yandex.solomon.ydb;

import java.util.List;

import javax.annotation.Nullable;

import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.settings.ReadTableSettings;
import com.yandex.ydb.table.values.TupleValue;

/**
 * @author Vladimir Gordiychuk
 */
public record SelectQuery(String query, Params params) {
    public static SelectQuery create(String tablePath, List<String> primaryKeys, ReadTableSettings settings) {
        StringBuilder builder = new StringBuilder();
        builder.append("--!syntax_v1\n");
        var params = appendDeclare(builder, primaryKeys, settings);
        builder.append("SELECT ");
        appendColumns(builder, settings.getColumns());
        appendFrom(builder, tablePath);
        appendWhere(builder, primaryKeys, settings);
        appendLimit(builder, settings);
        builder.append(";\n");
        return new SelectQuery(builder.toString(), params);
    }

    private static Params appendDeclare(StringBuilder builder, List<String> primaryKeys, ReadTableSettings settings) {
        var params = Params.create();
        var fromKey = settings.getFromKey();
        if (fromKey != null) {
            for (int index = 0; index < fromKey.size(); index++) {
                var value = fromKey.get(index);
                var param = "$from_" + primaryKeys.get(index);
                params.put(param, value);
                builder.append("DECLARE ").append(param).append(" AS ").append(value.getType()).append(";\n");
            }
        }

        var toKey = settings.getToKey();
        if (toKey != null) {
            for (int index = 0; index < toKey.size(); index++) {
                var value = toKey.get(index);
                var param = "$to_" + primaryKeys.get(index);
                params.put(param, value);
                builder.append("DECLARE ").append(param).append(" AS ").append(value.getType()).append(";\n");
            }
        }

        return params;
    }

    private static void appendColumns(StringBuilder builder, List<String> columns) {
        if (columns.isEmpty()) {
            builder.append("*");
        } else {
            for (int index = 0; index < columns.size() - 1; index++) {
                builder.append(columns.get(index)).append(", ");
            }
            builder.append(columns.get(columns.size() - 1));
        }
    }

    private static void appendFrom(StringBuilder builder, String tablePath) {
        builder.append(" FROM `").append(tablePath).append("`");
    }

    private static void appendWhere(StringBuilder builder, List<String> primaryKeys, ReadTableSettings settings) {
        if (isEmptyKey(settings.getFromKey()) && isEmptyKey(settings.getToKey())) {
            return;
        }

        builder.append(" WHERE ");
        var fromKey = settings.getFromKey();
        var toKey = settings.getToKey();
        if (!isEmptyKey(fromKey) && !isEmptyKey(toKey)) {
            builder.append("(");
            appendFromKey(builder, primaryKeys, fromKey, settings.isFromInclusive());
            builder.append(") AND (");
            appendToKey(builder, primaryKeys, toKey, settings.isToInclusive());
            builder.append(")");
        } else if (!isEmptyKey(fromKey)) {
            appendFromKey(builder, primaryKeys, fromKey, settings.isFromInclusive());
        } else if (!isEmptyKey(toKey)) {
            appendToKey(builder, primaryKeys, toKey, settings.isToInclusive());
        }
    }

    private static void appendFromKey(StringBuilder builder, List<String> primaryKeys, TupleValue fromKey, boolean inclusive) {
        // || (k1 > $k1)
        // || (k1 == $k1 and k2 > $k2)
        // || (k1 == $k1 and k2 == $k2 and k3 > $k3)
        for (int keySize = 1; keySize <= fromKey.size(); keySize++) {
            if (keySize > 1) {
                builder.append(" OR ");
                builder.append("(");
            }
            for (int index = 0; index < keySize; index++) {
                if (index > 0) {
                    builder.append(" AND ");
                }

                var column = primaryKeys.get(index);
                builder.append(column).append(" ");
                if (index + 1 < keySize) {
                    builder.append("==");
                } else if (inclusive && index + 1 == fromKey.size()) {
                    builder.append(">=");
                } else {
                    builder.append(">");
                }
                builder.append(" $").append("from_").append(column);
            }
            if (keySize > 1) {
                builder.append(")");
            }
        }
    }

    private static void appendToKey(StringBuilder builder, List<String> primaryKeys, TupleValue toKey, boolean inclusive) {
        // || (k1 < $k1)
        // || (k1 == $k1 and k2 < $k2)
        // || (k1 == $k1 and k2 == $k2 and k3 < $k3)
        for (int keySize = 1; keySize <= toKey.size(); keySize++) {
            if (keySize > 1) {
                builder.append(" OR ");
                builder.append("(");
            }
            for (int index = 0; index < keySize; index++) {
                if (index > 0) {
                    builder.append(" AND ");
                }

                var column = primaryKeys.get(index);
                builder.append(column).append(" ");
                if (index + 1 < keySize) {
                    builder.append("==");
                } else if (inclusive && index + 1 == toKey.size()) {
                    builder.append("<=");
                } else {
                    builder.append("<");
                }
                builder.append(" $").append("to_").append(column);
            }
            if (keySize > 1) {
                builder.append(")");
            }
        }
    }

    private static void appendLimit(StringBuilder builder, ReadTableSettings settings) {
        var limit = settings.getRowLimit();
        if (limit == 0) {
            return;
        }

        builder.append(" LIMIT ").append(Math.min(limit, 1000));
    }

    private static boolean isEmptyKey(@Nullable TupleValue key) {
        return key == null || key.isEmpty();
    }
}
