package ru.yandex.solomon.ydb;

import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.util.Optional;

import javax.annotation.Nullable;

import com.google.common.base.Preconditions;
import com.yandex.ydb.auth.tvm.YdbClientId;
import com.yandex.ydb.core.auth.AuthProvider;
import com.yandex.ydb.core.auth.NopAuthProvider;
import com.yandex.ydb.core.auth.TokenAuthProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.cloud.auth.token.TokenProvider;
import ru.yandex.cloud.token.IamTokenClient;
import ru.yandex.cloud.token.Jwt;
import ru.yandex.passport.tvmauth.NativeTvmClient;
import ru.yandex.passport.tvmauth.TvmApiSettings;
import ru.yandex.solomon.config.IamKeyJson;
import ru.yandex.solomon.config.protobuf.IamKeyAuthConfig;
import ru.yandex.solomon.config.protobuf.IamKeyJsonAuthConfig;
import ru.yandex.solomon.config.protobuf.TvmAuthConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.secrets.SecretProvider;
import ru.yandex.solomon.util.SolomonEnv;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbAuthProviders {
    private static final Logger logger = LoggerFactory.getLogger(YdbAuthProviders.class);

    @Nullable
    private final IamTokenClient iamTokenClient;
    private final ThreadPoolProvider threadPoolProvider;
    private final SecretProvider secretProvider;

    public YdbAuthProviders(
            @Nullable IamTokenClient iamTokenClient,
            ThreadPoolProvider threadPoolProvider,
            SecretProvider secretProvider)
    {
        this.iamTokenClient = iamTokenClient;
        this.threadPoolProvider = threadPoolProvider;
        this.secretProvider = secretProvider;
    }

    public AuthProvider tvm(TvmAuthConfig config) {
        Optional<String> clientSecret = secretProvider.getSecret(config.getClientSecret());
        if (config.getClientId() == 0 || clientSecret.isEmpty()) {
            return NopAuthProvider.INSTANCE;
        }

        int targetClientId = config.getTargetClientId() == 0
                ? YdbClientId.YDB.getId()
                : config.getTargetClientId();

        try (TvmApiSettings settings = TvmApiSettings.create()
                .setSelfTvmId(config.getClientId())
                .enableServiceTicketsFetchOptions(clientSecret.get(), new int[]{targetClientId}))
        {
            var client = NativeTvmClient.create(settings);
            return new TvmAuthProvider(client, targetClientId);
        }
    }

    public AuthProvider iam(IamKeyAuthConfig config) {
        ensureIamConfigured();
        if (!config.getPrivateKey().isEmpty()) {
            Optional<String> privateKey = secretProvider.getSecret(config.getPrivateKey());

            if (SolomonEnv.DEVELOPMENT.isActive() && privateKey.isEmpty()) {
                return makeFakeAuthProvider();
            }

            String privateKeyStr = privateKey.orElseThrow(() -> new RuntimeException("cannot create YDB IAM auth provider with empty private key"));
            return iam(config.getAccountId(), config.getKeyId(), privateKeyStr);
        } else if (!config.getKeyPath().isEmpty()) {
            Path keyFilePath = Path.of(config.getKeyPath());
            if (SolomonEnv.DEVELOPMENT.isActive() && !Files.exists(keyFilePath)) {
                return makeFakeAuthProvider();
            }

            return iam(config.getAccountId(), config.getKeyId(), keyFilePath);
        } else {
            throw new RuntimeException("invalid YDB IAM auth configuration. private_key or key_path cannot be empty");
        }
    }

    public AuthProvider iamKeyJson(IamKeyJsonAuthConfig config) {
        ensureIamConfigured();
        Preconditions.checkArgument(!config.getKeyData().isEmpty(), "IAM key json with empty key data");
        Optional<String> keyData = secretProvider.getSecret(config.getKeyData());
        if (SolomonEnv.DEVELOPMENT.isActive() && keyData.isEmpty()) {
            return makeFakeAuthProvider();
        }

        IamKeyJson iamKey = IamKeyJson.parseFromJson(
                keyData.orElseThrow(() -> new RuntimeException("cannot create YDB IAM auth provider with empty private key")));

        return iam(iamKey.getAccountId(), iamKey.getId(), iamKey.getPrivateKey());
    }

    private AuthProvider iam(String accountId, String keyId, String privateKey) {
        var jwtBuilder = Jwt.newBuilder()
                .withAccountId(accountId)
                .withKeyId(keyId)
                .withTtl(Duration.ofHours(1))
                .withPrivateKey(privateKey);

        var scheduler = threadPoolProvider.getSchedulerExecutorService();
        return new IamAuthProvider(TokenProvider.iam(iamTokenClient, jwtBuilder, scheduler));
    }

    private AuthProvider iam(String accountId, String keyId, Path privateKey) {
        var jwtBuilder = Jwt.newBuilder()
                .withAccountId(accountId)
                .withKeyId(keyId)
                .withTtl(Duration.ofHours(1))
                .withPrivateKey(privateKey);

        var scheduler = threadPoolProvider.getSchedulerExecutorService();
        return new IamAuthProvider(TokenProvider.iam(iamTokenClient, jwtBuilder, scheduler));
    }

    private void ensureIamConfigured() {
        if (iamTokenClient == null) {
            throw new RuntimeException("iam token client is not configured");
        }
    }

    private static AuthProvider makeFakeAuthProvider() {
        // only in development environment allowed to make fake token provide
        logger.warn("using fake token provider");
        return new TokenAuthProvider("fake-token");
    }

    private static class TvmAuthProvider implements AuthProvider {
        private final NativeTvmClient tvmClient;
        private final int targetClientId;

        public TvmAuthProvider(NativeTvmClient tvmClient, int targetClientId) {
            this.tvmClient = tvmClient;
            this.targetClientId = targetClientId;
        }

        @Override
        public String getToken() {
            return tvmClient.getServiceTicketFor(targetClientId);
        }
    }

    public static class IamAuthProvider implements AuthProvider {
        private final TokenProvider tokenProvider;

        public IamAuthProvider(TokenProvider tokenProvider) {
            this.tokenProvider = tokenProvider;
        }

        @Override
        public String getToken() {
            return tokenProvider.getToken();
        }
    }
}
