package ru.yandex.solomon.ydb;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;

import com.yandex.ydb.core.StatusCode;
import com.yandex.ydb.core.UnexpectedResultException;
import com.yandex.ydb.core.utils.Async;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.ExecuteDataQuerySettings;
import com.yandex.ydb.table.settings.ReadTableSettings;
import com.yandex.ydb.table.transaction.TxControl;
import com.yandex.ydb.table.values.TupleValue;
import com.yandex.ydb.table.values.Value;
import io.grpc.Context;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Objects.requireNonNull;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbReadTableTask {
    private static final Logger logger = LoggerFactory.getLogger(YdbReadTableTask.class);

    private final ReadTableSettings.Builder settings;
    private final SessionRetryContext retryCtx;
    private final String tablePath;
    private final YdbResultSetConsumer consumer;
    private final List<String> primaryKeyColumns;
    private final boolean continueReadAfterLimit;
    private final boolean useSelect;

    private final CompletableFuture<Void> doneFuture = new CompletableFuture<>();

    private TupleValue latestSeenPrimaryKey;
    private int readRows = 0;
    private boolean hasMore;

    private YdbReadTableTask(Builder builder) {
        this.settings = builder.settings;
        this.retryCtx = requireNonNull(builder.retryCtx, "retryCtx");
        this.tablePath = requireNonNull(builder.tablePath, "tablePath");
        this.consumer = requireNonNull(builder.consumer, "consumer");
        if (builder.continueReadAfterLimit && builder.primaryKeyColumns.isEmpty()) {
            throw new IllegalStateException("primary key columns not specified for paged read");
        }
        this.primaryKeyColumns = builder.primaryKeyColumns;
        this.continueReadAfterLimit = builder.continueReadAfterLimit;
        this.useSelect = builder.useSelect;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public CompletableFuture<Void> run() {
        // TODO: remote context fork after fix https://st.yandex-team.ru/YDBREQUESTS-335
        // It was necessary to avoid trigger kikimr bug when read table canceled after a few
        // millisecond because parent context was canceled by any reason, in case of alerting
        // it was assign shard with that loads async.
        var ctx = Context.current().fork();
        var prev = ctx.attach();
        try {
            if (continueReadAfterLimit) {
                continueReadSelect();
            } else {
                continueRead();
            }
            return doneFuture;
        } finally {
            ctx.detach(prev);
        }
    }

    private CompletableFuture<Void> waitReadyAccept() {
        return Async.safeCall(consumer::readyFuture)
                .exceptionally(e -> {
                    var code = StatusCode.CLIENT_INTERNAL_ERROR;
                    throw new UnexpectedResultException(code + ": ready future on consumer failed", code, e);
                });
    }

    private void continueReadSelect() {
        waitReadyAccept()
                .thenCompose(ignore -> readNextSelect())
                .thenRun(() -> {
                    if (hasMore) {
                        checkArgument(isNotEmpty(latestSeenPrimaryKey), "latest seen primary key is null, unable to continue read");
                        if (useSelect) {
                            continueReadSelect();
                        } else {
                            continueRead();
                        }
                    } else {
                        doneFuture.complete(null);
                    }
                })
                .exceptionally(e -> {
                    doneFuture.completeExceptionally(e);
                    return null;
                });
    }

    private void continueRead() {
        waitReadyAccept()
                .thenCompose(ignore -> readNext())
                .thenRun(() -> {
                    if (hasMore && continueReadAfterLimit) {
                        checkArgument(isNotEmpty(latestSeenPrimaryKey), "latest seen primary key is null, unable to continue read");
                        continueRead();
                    } else {
                        doneFuture.complete(null);
                    }
                })
                .exceptionally(e -> {
                    doneFuture.completeExceptionally(e);
                    return null;
                });
    }

    private boolean isNotEmpty(@Nullable TupleValue value) {
        return !isEmpty(value);
    }

    private boolean isEmpty(@Nullable TupleValue value) {
        return value == null || value.isEmpty();
    }

    private boolean isKeyRangeEmpty(ReadTableSettings settings) {
        var toKey = settings.getToKey();
        if (isEmpty(toKey)) {
            return false;
        }

        var fromKey = settings.getFromKey();
        if (isEmpty(fromKey)) {
            return false;
        }

        return toKey.toPb().equals(fromKey.toPb());
    }

    private ReadTableSettings actualSettings() {
        var fromKeyExclusive = latestSeenPrimaryKey;
        if (isNotEmpty(fromKeyExclusive)) {
            settings.fromKeyExclusive(fromKeyExclusive);
        }

        return settings.build();
    }

    private CompletableFuture<Void> readNextSelect() {
        try {
            readRows = 0;
            return retryCtx.supplyResult(session -> {
                var settings = actualSettings();
                if (isEmpty(settings.getFromKey())) {
                    consumer.restart();
                }

                hasMore = false;
                var selectQuery = SelectQuery.create(tablePath, primaryKeyColumns, settings);
                var selectSettings = new ExecuteDataQuerySettings()
                        .keepInQueryCache();
                return session.executeDataQuery(selectQuery.query(), TxControl.onlineRo(), selectQuery.params(), selectSettings)
                        .whenComplete((r, e) -> {
                            if (e != null) {
                                logger.warn("select table {} with with range {}:{} failed", tablePath, settings.getFromKey(), settings.getToKey(), e);
                            } else if (!r.isSuccess()) {
                                logger.warn("select table {} with settings {}:{} failed {}", tablePath, settings.getFromKey(), settings.getToKey(), r.toStatus());
                            }
                        });
            }).thenAccept(r -> {
                var dataQuery = r.expect("can't read as select " + tablePath);
                var rs = dataQuery.getResultSet(0);
                onResultSet(rs);
                if (rs.getRowCount() > 0 && !hasMore) {
                    var settings = actualSettings();
                    hasMore = readRows >= Math.min(settings.getRowLimit(), 1000) && !isKeyRangeEmpty(settings);
                }
            });
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    private CompletableFuture<Void> readNext() {
        try {
            readRows = 0;
            return retryCtx.supplyStatus(session -> {
                var settings = actualSettings();
                if (isEmpty(settings.getFromKey())) {
                    consumer.restart();
                }

                hasMore = false;
                logger.info("read table {} with range from:{}(inclusive:{}) to:{}(inclusive:{})",
                        tablePath, settings.getFromKey(), settings.isFromInclusive(),
                        settings.getToKey(), settings.isToInclusive());
                return session.readTable(tablePath, settings, this::onResultSet)
                        .whenComplete((r, e) -> {
                            if (e != null) {
                                logger.warn("read table {} with with range {}:{} failed", tablePath, settings.getFromKey(), settings.getToKey(), e);
                            } else if (!r.isSuccess()) {
                                logger.warn("read table {} with settings {}:{} failed {}", tablePath, settings.getFromKey(), settings.getToKey(), r);
                            } else if (readRows < settings.getRowLimit()) {
                                hasMore = false;
                            }
                        });
            }).thenAccept(status -> status.expect("can't read " + tablePath));
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    private void onResultSet(ResultSetReader rs) {
        try {
            if (rs.getRowCount() > 0) {
                rs.setRowIndex(rs.getRowCount() - 1);
                List<Value> values = new ArrayList<>(primaryKeyColumns.size());
                for (var columnName : primaryKeyColumns) {
                    values.add(rs.getColumn(columnName).getValue());
                }
                latestSeenPrimaryKey = TupleValue.of(values);
                rs.setRowIndex(0);
            }

            hasMore = rs.isTruncated();
            readRows += rs.getRowCount();
            consumer.accept(rs);
        } catch (Throwable e) {
            var code = StatusCode.CLIENT_INTERNAL_ERROR;
            throw new UnexpectedResultException(code + ": process ResultSet failed on client side", code, e);
        }
    }

    public static class Builder {
        private final ReadTableSettings.Builder settings = ReadTableSettings.newBuilder()
                .orderedRead(true)
                .timeout(1, TimeUnit.MINUTES);

        private SessionRetryContext retryCtx;
        private String tablePath;
        private YdbResultSetConsumer consumer;
        private List<String> primaryKeyColumns = List.of();
        private boolean continueReadAfterLimit;
        private boolean useSelect;

        private Builder() {
        }

        public Builder fromKey(TupleValue value, boolean inclusive) {
            settings.fromKey(value, inclusive);
            return this;
        }

        public Builder rowLimit(int limit) {
            settings.rowLimit(limit);
            return this;
        }

        public Builder continueReadAfterLimit(boolean value) {
            continueReadAfterLimit = value;
            return this;
        }

        public Builder toKey(TupleValue value, boolean inclusive) {
            settings.toKey(value, inclusive);
            return this;
        }

        public Builder timeout(long duration, TimeUnit unit) {
            settings.timeout(duration, unit);
            return this;
        }

        public Builder columns(List<String> columns) {
            settings.columns(columns);
            return this;
        }

        public Builder primaryKeys(List<String> columns) {
            primaryKeyColumns = List.copyOf(columns);
            return this;
        }

        public Builder retryContext(SessionRetryContext retryCtx) {
            this.retryCtx = retryCtx;
            return this;
        }

        public Builder tablePath(String path) {
            this.tablePath = path;
            return this;
        }

        public Builder consumer(YdbResultSetConsumer consumer) {
            this.consumer = consumer;
            return this;
        }

        public Builder useSelect(boolean useSelect) {
            this.useSelect = useSelect;
            return this;
        }

        public YdbReadTableTask build() {
            return new YdbReadTableTask(this);
        }
    }
}
