package ru.yandex.solomon.ydb;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;

import com.yandex.ydb.table.result.ResultSetReader;
import io.grpc.Status;

import ru.yandex.misc.actor.ActorRunner;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbResultSetAsyncConsumer implements YdbResultSetConsumer {
    private final YdbResultSetConsumer consumer;
    private final ActorRunner actor;
    private final int maxQueueSize;
    private final AtomicInteger queueSize = new AtomicInteger();
    private final ConcurrentLinkedQueue<Event> events = new ConcurrentLinkedQueue<>();
    private final CompletableFuture<Void> done = new CompletableFuture<>();
    private volatile CompletableFuture<Void> queueFree = CompletableFuture.completedFuture(null);

    public YdbResultSetAsyncConsumer(YdbResultSetConsumer consumer, int maxQueueSize, Executor executor) {
        this.consumer = consumer;
        this.maxQueueSize = maxQueueSize;
        this.actor = new ActorRunner(this::act, executor);
    }

    @Override
    public void accept(ResultSetReader resultSet) {
        if (done.isDone()) {
            throw Status.FAILED_PRECONDITION
                    .withDescription("already done")
                    .asRuntimeException();
        }

        int size = queueSize.addAndGet(resultSet.getRowCount());
        if (size >= maxQueueSize && queueFree.isDone()) {
            queueFree = new CompletableFuture<>();
        }
        events.add(new Data(resultSet));
        actor.schedule();
    }

    @Override
    public void restart() {
        events.add(new Restart());
        actor.schedule();
    }

    private void act() {
        Event event;
        while ((event = events.poll()) != null) {
            try {
                if (done.isDone()) {
                    continue;
                }

                if (event instanceof Done) {
                    done.complete(null);
                } else if (event instanceof Data d) {
                    queueSize.addAndGet(-d.rs.getRowCount());
                    consumer.accept(d.rs);
                } else if (event instanceof Restart) {
                    consumer.restart();
                }
            } catch (Throwable e) {
                done.completeExceptionally(e);
                queueFree.completeExceptionally(e);
            } finally {
                int size = queueSize.get();
                if (size < maxQueueSize) {
                    queueFree.complete(null);
                }
            }
        }
    }

    public CompletableFuture<Void> done() {
        events.add(new Done());
        actor.schedule();
        return done;
    }

    @Override
    public CompletableFuture<Void> readyFuture() {
        return queueFree;
    }

    sealed interface Event {};
    record Done() implements Event { }
    record Restart() implements Event { }
    record Data(ResultSetReader rs) implements Event { }
}
