package ru.yandex.solomon.ydb;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;

import com.google.common.util.concurrent.MoreExecutors;
import com.yandex.ydb.core.Result;
import com.yandex.ydb.core.Status;
import com.yandex.ydb.table.Session;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.CreateTableSettings;
import com.yandex.ydb.table.settings.ExecuteDataQuerySettings;
import com.yandex.ydb.table.settings.ReadTableSettings;
import com.yandex.ydb.table.transaction.TxControl;
import io.grpc.Context;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;
import ru.yandex.solomon.ydb.page.TokenBasePage;
import ru.yandex.solomon.ydb.page.TokenPageOptions;

import static com.google.common.base.Preconditions.checkState;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;


/**
 * @author Sergey Polovko
 */
public abstract class YdbTable<Id extends Comparable<Id>, Entity> {
    private static final Logger logger = LoggerFactory.getLogger(YdbTable.class);
    private static final Duration FIND_ALL_TIMEOUT = Duration.ofSeconds(30);

    private final SessionRetryContext retryCtx;
    private final String path;
    private final Executor executor;
    private volatile boolean useSelect;

    protected YdbTable(TableClient tableClient, String path) {
        this(tableClient, path, MoreExecutors.directExecutor());
    }

    protected YdbTable(TableClient tableClient, String path, Executor executor) {
        this.retryCtx = SessionRetryContext.create(tableClient)
                .maxRetries(10)
                .executor(executor)
                .build();
        this.path = path;
        this.executor = executor;
    }

    @ManagerMethod
    void useSelect(boolean value) {
        this.useSelect = value;
    }

    public String getPath() {
        return path;
    }

    protected abstract TableDescription description();
    protected abstract Id getId(Entity entity);
    protected abstract Params toParams(Entity entity);
    protected abstract Entity mapFull(ResultSetReader resultSet);
    protected abstract Entity mapPartial(ResultSetReader resultSet);

    // table path -> table description
    protected Map<String, TableDescription> indexes() {
        return Collections.emptyMap();
    }

    protected CreateTableSettings createTableSettings(CreateTableSettings settings) {
        return settings;
    }

    public CompletableFuture<Boolean> insertOne(String query, Entity entity) {
        try {
            TxControl txControl = TxControl.serializableRw().setCommitTx(true);
            ExecuteDataQuerySettings settings = new ExecuteDataQuerySettings()
                .keepInQueryCache();
            Params params = toParams(entity);

            return retryCtx.supplyResult(session -> session.executeDataQuery(query, txControl, params, settings))
                .thenApply(result -> {
                    if (!result.isSuccess()) {
                        logger.debug("Unsuccessful insert {}: {}", entity, result);
                    }

                    switch (result.getCode()) {
                        case SUCCESS: return Boolean.TRUE;
                        case PRECONDITION_FAILED: return Boolean.FALSE;
                    }
                    result.expect("cannot insert " + entity.getClass().getSimpleName());
                    return null; // must never happen
                });
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    public CompletableFuture<Optional<Entity>> updateOne(String query, Entity entity) {
        try {
            return updateOne(query, toParams(entity));
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    public CompletableFuture<Void> upsertOne(String query, Entity entity) {
        try {
            return executeAndExpectSuccess(query, toParams(entity)).thenApply(v -> null);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    public CompletableFuture<Optional<Entity>> updateOne(String query, Params params) {
        return executeAndExpectSuccess(query, params)
            .thenApply(result -> {
                if (result.getResultSetCount() == 0) {
                    return Optional.empty();
                }
                ResultSetReader resultSet = result.getResultSet(0);
                if (!resultSet.next()) {
                    return Optional.empty();
                }
                return Optional.of(mapFull(resultSet));
            });
    }

    public CompletableFuture<Optional<Entity>> queryOne(String query, Params params) {
        return executeAndExpectSuccess(query, params)
            .thenApply(result -> {
                ResultSetReader resultSet = result.getResultSet(0);
                if (resultSet.next()) {
                    return Optional.of(mapFull(resultSet));
                }
                return Optional.empty();
            });
    }

    public CompletableFuture<Integer> queryCount(String query, Params params) {
        return executeAndExpectSuccess(query, params)
            .thenApply(result -> result.getResultSetCount());
    }

    public CompletableFuture<List<Entity>> queryAll() {
        return queryAll(r -> true);
    }

    public CompletableFuture<List<Entity>> queryAll(Predicate<ResultSetReader> predicate) {
        var result = new ArrayList<Entity>();
        var consumer = new YdbResultSetAsyncConsumer(new YdbResultSetConsumer() {
            @Override
            public void restart() {
                result.clear();
            }

            @Override
            public void accept(ResultSetReader resultSet) {
                result.ensureCapacity(resultSet.getRowCount());
                while (resultSet.next()) {
                    if (predicate.test(resultSet)) {
                        result.add(mapFull(resultSet));
                    }
                }
            }
        }, 300_000, executor);

        var task = readTable()
                .primaryKeys(description().getPrimaryKeys())
                .continueReadAfterLimit(true)
                .rowLimit(300_000)
                .consumer(consumer)
                .build();

        return task.run()
                .thenCompose(ignore -> consumer.done())
                .thenApply(ignore -> result);
    }

    public CompletableFuture<List<Entity>> queryAll(Predicate<ResultSetReader> predicate, ReadTableSettings settings) {
        class ToListConsumer implements Consumer<ResultSetReader> {
            private final ArrayList<Entity> entities = new ArrayList<>();

            @Override
            public void accept(ResultSetReader resultSet) {
                entities.ensureCapacity(resultSet.getRowCount());
                // TODO: can be optimized by getting column indexes outside of the loop body
                while (resultSet.next()) {
                    if (predicate.test(resultSet)) {
                        entities.add(mapFull(resultSet));
                    }
                }
            }

            private Result<List<Entity>> toResult(Status s) {
                return s.isSuccess() ? Result.success(entities) : Result.fail(s);
            }
        }

        try {
            return retryCtx.supplyResult(session -> {
                ToListConsumer consumer = new ToListConsumer();
                // TODO: remote context fork after fix https://st.yandex-team.ru/YDBREQUESTS-335
                // It was necessary to avoid trigger kikimr bug when read table canceled after a few
                // millisecond because parent context was canceled by any reason, in case of alerting
                // it was assign shard with that loads async.
                var ctx = Context.current().fork();
                var prev = ctx.attach();
                try {
                    return session.readTable(path, settings, consumer)
                        .thenApply(consumer::toResult)
                        .whenComplete((r, e) -> {
                            if (e != null) {
                                logger.warn("read table {} with with range {}:{} failed", path, settings.getFromKey(), settings.getToKey(), e);
                            } else if (!r.isSuccess()) {
                                logger.warn("read table {} with settings {}:{} failed {}", path, settings.getFromKey(), settings.getToKey(), r);
                            }
                        });
                } finally {
                    ctx.detach(prev);
                }
            })
                .thenApply(result -> result.expect("cannot read table " + path));
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    public YdbReadTableTask.Builder readTable() {
        return YdbReadTableTask.newBuilder()
                .useSelect(useSelect)
                .tablePath(path)
                .retryContext(retryCtx);
    }

    public CompletableFuture<List<Entity>> queryList(String query, Params params) {
        return queryList(query, params, false);
    }

    public CompletableFuture<List<Entity>> queryListFull(String query, Params params) {
        return queryList(query, params, true);
    }

    private CompletableFuture<List<Entity>> queryList(String query, Params params, boolean mapFull) {
        return queryList(query, params, mapFull, (ignore, entities) -> entities);
    }

    private <R> CompletableFuture<R> queryList(
        String query,
        Params params,
        boolean mapFull,
        BiFunction<ResultSetReader, List<Entity>, R> finisher)
    {
        return executeAndExpectSuccess(query, params)
            .thenApply(result -> {
                ResultSetReader resultSet = result.getResultSet(0);
                if (resultSet.getRowCount() == 0) {
                    return finisher.apply(resultSet, List.of());
                }
                ArrayList<Entity> entities = new ArrayList<>(resultSet.getRowCount());
                if (mapFull) {
                    while (resultSet.next()) {
                        entities.add(mapFull(resultSet));
                    }
                } else {
                    while (resultSet.next()) {
                        entities.add(mapPartial(resultSet));
                    }
                }
                return finisher.apply(resultSet, entities);
            });
    }

    public CompletableFuture<PagedResult<Entity>> queryPage(
        Params params,
        PageOptions pageOpts,
        Function<PageOptions, String> queryBuilder)
    {
        try {
            if (pageOpts.isLimited()) {
                String query = queryBuilder.apply(pageOpts);
                return queryPage(query, params, pageOpts);
            }
            AsyncFinder<Id, Entity> finder = new AsyncFinder<>(this, params, queryBuilder);
            finder.findAll(Instant.now().plus(FIND_ALL_TIMEOUT));
            return finder.getFuture();
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    public CompletableFuture<TokenBasePage<Entity>> queryPage(
        Params params,
        TokenPageOptions pageOpts,
        Function<TokenPageOptions, String> queryBuilder)
    {
        try {
            return queryPage(
                queryBuilder.apply(pageOpts),
                params,
                pageOpts,
                ignore -> String.valueOf(pageOpts.getSize() + pageOpts.getOffset()));
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    public CompletableFuture<TokenBasePage<Entity>> queryPage(
        String query,
        Params params,
        TokenPageOptions pageOpts,
        Function<Entity, String> nextPageTokenFn)
    {
        return queryList(query, params, true, (resultSet, entities) -> {
            final List<Entity> result;
            final String nextPageToken;

            var hasMore = entities.size() > pageOpts.getSize();
            if (hasMore || resultSet.isTruncated()) {
                result = hasMore
                    ? entities.subList(0, pageOpts.getSize())
                    : entities;
                nextPageToken = nextPageTokenFn.apply(result.get(result.size() - 1));
            } else {
                result = entities;
                nextPageToken = "";
            }

            return new TokenBasePage<>(result, nextPageToken);
        });
    }

    public CompletableFuture<PagedResult<Entity>> queryPage(String query, Params params, PageOptions pageOpts) {
        return executeAndExpectSuccess(query, params)
            .thenApply(result -> {
                checkState(result.getResultSetCount() == 2, "expected 2 result sets");

                // (1) get total entities count
                ResultSetReader countResult = result.getResultSet(0);
                checkState(countResult.getRowCount() == 1, "expected result set with single row");
                countResult.next();
                int totalCount = (int) countResult.getColumn(0).getUint64(); // TODO: change type to Int32

                // (2) get entities
                ResultSetReader dataResult = result.getResultSet(1);
                List<Entity> entities = new ArrayList<>(dataResult.getRowCount());
                while (dataResult.next()) {
                    entities.add(mapPartial(dataResult));
                }
                return PagedResult.of(entities, pageOpts, totalCount);
            });
    }

    public CompletableFuture<Boolean> queryBool(String query, Params params) {
        return executeAndExpectSuccess(query, params)
            .thenApply(result -> {
                ResultSetReader resultSet = result.getResultSet(0);
                if (resultSet.next()) {
                    return resultSet.getColumn(0).getBool();
                }
                return Boolean.FALSE;
            });
    }

    public CompletableFuture<Void> queryVoid(String query, Params params) {
        return executeAndExpectSuccess(query, params)
            .thenAccept(result -> {});
    }

    public CompletableFuture<Result<DataQueryResult>> execute(String query, Params params) {
        try {
            ExecuteDataQuerySettings settings = new ExecuteDataQuerySettings()
                .keepInQueryCache();
            return retryCtx.supplyResult(session -> session.executeDataQuery(query, TxControl.serializableRw(), params, settings));
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    public CompletableFuture<DataQueryResult> executeAndExpectSuccess(String query, Params params) {
        try {
            return execute(query, params)
                .thenApply(result -> {
                    if (!result.isSuccess()) {
                        logger.error("cannot execute query {}:\n{}", result, query);
                    }
                    return result.expect("cannot execute query");
                });
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    public SessionRetryContext retryCtx() {
        return retryCtx;
    }

    public CompletableFuture<Void> create() {
        CreateTableSettings tableSettings = createTableSettings(
            new CreateTableSettings().setTimeout(10, TimeUnit.SECONDS));
        TableDescription tableDescription = description();
        Map<String, TableDescription> indexes = indexes();

        Function<Session, CompletableFuture<Status>> doCreate = (session) -> {
            var indexSettings = new CreateTableSettings().setTimeout(10, TimeUnit.SECONDS);
            var future = session.createTable(path, tableDescription, tableSettings);
            for (Map.Entry<String, TableDescription> e : indexes.entrySet()) {
                future = future.thenCompose(s -> {
                    if (!s.isSuccess()) {
                        return completedFuture(s);
                    }
                    return session.createTable(e.getKey(), e.getValue(), indexSettings);
                });
            }
            return future;
        };
        return retryCtx.supplyStatus(doCreate)
            .thenAccept(status -> {
                status.expect("cannot create " + path + " table");
            });
    }

    public CompletableFuture<Void> drop() {
        Map<String, TableDescription> indexes = indexes();
        Function<Session, CompletableFuture<Status>> doDrop = (session) -> {
            var future = session.dropTable(path);
            for (String indexPath : indexes.keySet()) {
                future = future.thenCompose(s -> {
                    if (!s.isSuccess()) {
                        return completedFuture(s);
                    }
                    return session.dropTable(indexPath);
                });
            }
            return future;
        };
        return retryCtx.supplyStatus(doDrop)
            .thenAccept(status -> {
                status.expect("cannot drop " + path + " table");
            });
    }
}
