package ru.yandex.solomon.ydb.page;


import java.util.Collections;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;


/**
 * @author Sergey Polovko
 */
public class PagedResult<T> {

    private static final PagedResult<?> EMPTY = new PagedResult<>(
        Collections.emptyList(), 0, 0, 0, 0);

    private final List<T> result;
    private final int totalCount;
    private final int pagesCount;
    private final int pageSize;
    private final int currentPage;


    public PagedResult(List<T> result, int pagesCount, int totalCount, int pageSize, int currentPage) {
        this.result = result;
        this.pagesCount = pagesCount;
        this.totalCount = totalCount;
        this.pageSize = pageSize;
        this.currentPage = currentPage;
    }

    public static <U> PagedResult<U> of(List<U> result, PageOptions pageOpts, int totalCount) {
        int pagesCount = 1;
        int pageSize = pageOpts.getSize();
        int currentPage = pageOpts.getCurrent();

        if (pageOpts.isLimited()) {
            pagesCount = (int) Math.ceil((double) totalCount / pageOpts.getSize());
        }

        return new PagedResult<>(result, pagesCount, totalCount, pageSize, currentPage);
    }

    @SuppressWarnings("unchecked")
    public static <U> PagedResult<U> empty() {
        return (PagedResult<U>) EMPTY;
    }

    public List<T> getResult() {
        return result;
    }

    public int getPagesCount() {
        return pagesCount;
    }

    public int getTotalCount() {
        return totalCount;
    }

    public int getPageSize() {
        return pageSize;
    }

    public int getCurrentPage() {
        return currentPage;
    }

    public <U> PagedResult<U> map(Function<? super T, U> fn) {
        List<U> mappedResult = result.stream().map(fn).collect(Collectors.toList());
        return new PagedResult<>(mappedResult, pagesCount, totalCount, pageSize, currentPage);
    }
}
