package ru.yandex.solomon.ydb.page;

import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class TokenBasePage<T> {
    private static final TokenBasePage<?> EMPTY = new TokenBasePage<>(List.of(), "");
    private final List<T> items;
    private final String nextPageToken;

    public TokenBasePage(List<T> items, String nextPageToken) {
        this.items = items;
        this.nextPageToken = nextPageToken;
    }

    public static <T> TokenBasePage<T> of(List<T> items, TokenPageOptions pageOptions) {
        final String nextPageToken;
        final List<T> result;

        if (items.size() > pageOptions.getSize()) {
            result = items.subList(0, pageOptions.getSize());
            nextPageToken = String.valueOf(pageOptions.getOffset() + pageOptions.getSize());
        } else {
            result = items;
            nextPageToken = "";
        }

        return new TokenBasePage<>(result, nextPageToken);
    }

    public static <T> TokenBasePage<T> empty() {
        return (TokenBasePage<T>) EMPTY;
    }

    public List<T> getItems() {
        return items;
    }

    public String getNextPageToken() {
        return nextPageToken;
    }

    public <U> TokenBasePage<U> map(Function<T, U> map) {
        return new TokenBasePage<>(items.stream().map(map).collect(Collectors.toList()), nextPageToken);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        TokenBasePage<?> that = (TokenBasePage<?>) o;

        if (!nextPageToken.equals(that.nextPageToken)) {
            return false;
        }
        return items.equals(that.items);
    }

    @Override
    public int hashCode() {
        int result = items.hashCode();
        result = 31 * result + nextPageToken.hashCode();
        return result;
    }

    @Override
    public String toString() {
        return "TokenBasePage{" +
                "items=" + items +
                ", nextPageToken=" + nextPageToken +
                '}';
    }
}
