package ru.yandex.solomon.ydb;

import java.util.List;

import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.settings.ReadTableSettings;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.TupleValue;
import org.junit.Test;

import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class SelectQueryTest {

    @Test
    public void selectAll() {
        var selectQuery = SelectQuery.create("/root/test", List.of(), ReadTableSettings.newBuilder().build());
        var expected = """
                --!syntax_v1
                SELECT * FROM `/root/test`;
                """;

        assertEquals(expected, selectQuery.query());
        assertEquals(Params.empty().toPb(), selectQuery.params().toPb());
    }

    @Test
    public void selectColumns() {
        var settings = ReadTableSettings.newBuilder()
                .columns("k1", "k2", "value")
                .build();
        var selectQuery = SelectQuery.create("/root/test", List.of(), settings);
        var expected = """
                --!syntax_v1
                SELECT k1, k2, value FROM `/root/test`;
                """;

        assertEquals(expected, selectQuery.query());
        assertEquals(Params.empty().toPb(), selectQuery.params().toPb());
    }

    @Test
    public void fromKeyExclusive() {
        var settings = ReadTableSettings.newBuilder()
                .fromKey(TupleValue.of(PrimitiveValue.utf8("alice").makeOptional(), PrimitiveValue.utf8("001").makeOptional()), false)
                .build();
        var selectQuery = SelectQuery.create("/root/test", List.of("k1", "k2"), settings);
        var expected = """
                --!syntax_v1
                DECLARE $from_k1 AS Utf8?;
                DECLARE $from_k2 AS Utf8?;
                SELECT * FROM `/root/test` WHERE k1 > $from_k1 OR (k1 == $from_k1 AND k2 > $from_k2);
                """;

        assertEquals(expected, selectQuery.query());

        var expectParams = Params.of(
                "$from_k1", PrimitiveValue.utf8("alice").makeOptional(),
                "$from_k2", PrimitiveValue.utf8("001").makeOptional());
        assertEquals(expectParams.toPb(), selectQuery.params().toPb());
    }

    @Test
    public void fromKeyInclusive() {
        var settings = ReadTableSettings.newBuilder()
                .fromKey(TupleValue.of(PrimitiveValue.utf8("alice").makeOptional(), PrimitiveValue.utf8("001").makeOptional()), true)
                .build();
        var selectQuery = SelectQuery.create("/root/test", List.of("k1", "k2"), settings);
        var expected = """
                --!syntax_v1
                DECLARE $from_k1 AS Utf8?;
                DECLARE $from_k2 AS Utf8?;
                SELECT * FROM `/root/test` WHERE k1 > $from_k1 OR (k1 == $from_k1 AND k2 >= $from_k2);
                """;

        assertEquals(expected, selectQuery.query());

        var expectParams = Params.of(
                "$from_k1", PrimitiveValue.utf8("alice").makeOptional(),
                "$from_k2", PrimitiveValue.utf8("001").makeOptional());
        assertEquals(expectParams.toPb(), selectQuery.params().toPb());
    }

    @Test
    public void toKeyExclusive() {
        var settings = ReadTableSettings.newBuilder()
                .toKey(TupleValue.of(PrimitiveValue.utf8("bob").makeOptional(), PrimitiveValue.utf8("002").makeOptional()), false)
                .build();
        var selectQuery = SelectQuery.create("/root/test", List.of("k1", "k2"), settings);
        var expected = """
                --!syntax_v1
                DECLARE $to_k1 AS Utf8?;
                DECLARE $to_k2 AS Utf8?;
                SELECT * FROM `/root/test` WHERE k1 < $to_k1 OR (k1 == $to_k1 AND k2 < $to_k2);
                """;

        assertEquals(expected, selectQuery.query());

        var expectParams = Params.of(
                "$to_k1", PrimitiveValue.utf8("bob").makeOptional(),
                "$to_k2", PrimitiveValue.utf8("002").makeOptional());
        assertEquals(expectParams.toPb(), selectQuery.params().toPb());
    }

    @Test
    public void toKeyInclusive() {
        var settings = ReadTableSettings.newBuilder()
                .toKey(TupleValue.of(PrimitiveValue.utf8("bob").makeOptional(), PrimitiveValue.utf8("002").makeOptional()), true)
                .build();
        var selectQuery = SelectQuery.create("/root/test", List.of("k1", "k2"), settings);
        var expected = """
                --!syntax_v1
                DECLARE $to_k1 AS Utf8?;
                DECLARE $to_k2 AS Utf8?;
                SELECT * FROM `/root/test` WHERE k1 < $to_k1 OR (k1 == $to_k1 AND k2 <= $to_k2);
                """;

        assertEquals(expected, selectQuery.query());

        var expectParams = Params.of(
                "$to_k1", PrimitiveValue.utf8("bob").makeOptional(),
                "$to_k2", PrimitiveValue.utf8("002").makeOptional());
        assertEquals(expectParams.toPb(), selectQuery.params().toPb());
    }

    @Test
    public void selectLimit() {
        var settings = ReadTableSettings.newBuilder()
                .rowLimit(2)
                .build();

        var selectQuery = SelectQuery.create("/root/test", List.of(), settings);
        var expected = """
                --!syntax_v1
                SELECT * FROM `/root/test` LIMIT 2;
                """;

        assertEquals(expected, selectQuery.query());
        assertEquals(Params.empty().toPb(), selectQuery.params().toPb());
    }
}
