package ru.yandex.solomon.co2;

import java.util.List;
import java.util.Optional;
import java.util.Queue;
import java.util.stream.Collectors;

import org.hid4java.HidDevice;
import org.hid4java.HidManager;
import org.hid4java.HidServices;

/**
 * This code is derived on https://github.com/maizy/ambient7/tree/master/mt8057-agent
 *
 * @author (acrually, @translator) Maksim Leonov (nohttp@)
 */
public class Co2 {
    enum States {
        UNKNOWN,
        WAIT,
        INIT,
        READ,
        /**/
    }

    private static final short VENDOR_ID = 0x04d9;
    private static final short PRODUCT_ID = (short) 0xa052;

    private final HidServices hidServices = HidManager.getHidServices();
    private Optional<HidDevice> device = Optional.empty();

    private States state = States.UNKNOWN;
    private final Queue<MetricMessage> queue;

    public Co2(Queue<MetricMessage> queue) {
        this.queue = queue;
        hidServices.start();
    }

    private Optional<HidDevice> findDevice() {
        List<HidDevice> attached = hidServices.getAttachedHidDevices();
        List<HidDevice> matched = attached.stream()
                .filter(this::isMatchedDevice)
                .collect(Collectors.toList());
        System.err.println("findDevice: " + matched.size() + " found");
        if (matched.size() > 1) {
            System.err.println("More than one matched USB HID devices");
            return Optional.empty();
        } else if (matched.isEmpty()) {
            return Optional.empty();
        } else {
//            // hid4java works only if get device like this
//            // device from getAttachedHidDevices doesn't work properly
            System.out.println(matched.get(0).open());
            return Optional.of(matched.get(0));//hidServices.getHidDevice(VENDOR_ID, PRODUCT_ID, null));
        }
    }

    private boolean isMatchedDevice(HidDevice device) {
        return device.getVendorId() == VENDOR_ID && device.getProductId() == PRODUCT_ID;
    }

    void run() throws InterruptedException {
        state = States.WAIT;
        device = findDevice();
//        if (device.isPresent()) {
////            queue.add(DeviceUp(currentNanoTime()))
//        }
        long standardDelay = 500;
        long delay = standardDelay; // ms
        while (true) {
            switch (state) {
                case UNKNOWN:
                    closeDevice();
                    state = States.WAIT;
                    delay = standardDelay * 2;
                    break;
                case WAIT:
                    if (!device.isPresent()) {
                        device = findDevice();
                    }
                    if (device.isPresent()) {
                        state = States.INIT;
                        delay = 50;
                    } else {
                        delay = standardDelay * 2;
                    }
                    break;
                case INIT:
                    if (device.isPresent() && initDevice()) {
                        state = States.READ;
                        delay = 50;
                    } else {
                        System.err.println("init error");
                        closeDevice();
                        state = States.WAIT;
                        delay = standardDelay * 2;
                    }
                    break;
                case READ:
                    if (!device.isPresent() || !readData()) {
                        System.err.println("read error");
                        closeDevice();
                        state = States.WAIT;
                        delay = standardDelay * 2;
                    } else {
                        delay = standardDelay;
                    }
                    break;
            }
            Thread.sleep(delay);
        }
    }

    private boolean initDevice() {
        if (device.isPresent()) {
            int packageSize = 8;
            byte[] magicTable = new byte[packageSize]; // send zero table
            return device.get().sendFeatureReport(magicTable, (byte) 0x0) == packageSize + 1;
        } else {
            return false;
        }
    }

    private synchronized void closeDevice() {
        device.filter(HidDevice::isOpen).ifPresent(HidDevice::close);
        device = Optional.empty();
    }

    private boolean readData() {
        byte[] data = new byte[8];
        if (device.isPresent()) {
            switch (device.get().read(data, 5000)) {
                case -1: // unknown error
                    return false;
                case 0:  // no data to read
                    return true;
                default:
                    try {
                        byte[] decoded = MessageDecoder.decode(data);
                        if (!MessageDecoder.checkCRC(decoded)) {
                            System.err.println("Bad CRC!");
                            printArray("data", data);
                            printArray("decoded", decoded);
                        } else {
                            MessageDecoder.parseValue(decoded).ifPresent(queue::add);
                        }
                    } catch (Exception e) {
                        System.err.println("Decode problem");
                    }
            }
            return true;
        } else {
            return false;
        }
    }

    private static void printArray(String prefix, byte[] data) {
        System.err.print(prefix + ": ");
        for (byte b: data) {
            String repr = Integer.toHexString(b&0xff);
            if (repr.length() == 1) {
                repr = "0" + repr;
            }
            System.err.print(repr + " ");
        }
        System.err.println();
    }
}
