package ru.yandex.solomon.co2;

import java.time.Instant;
import java.util.Arrays;
import java.util.Optional;

/**
 * @author Maksim Leonov (nohttp@)
 */
public class MessageDecoder {
    private static byte[] MAGIC_WORD = new byte[]{0x48, 0x74, 0x65, 0x6d, 0x70, 0x39, 0x39, 0x65}; // Htemp99e
    private static byte[] MAGIC_WORD_SHIFTED = new byte[]{(byte) 0x84, 0x47, 0x56, (byte) 0xd6, 0x07, (byte) 0x93, (byte) 0x93, 0x56};
    private static final byte CO2_CODE = 0x50;
    private static final byte TEMP_CODE = 0x42;
    private static double ABS_ZERO_TEMP = 273.15;
    private static double TEMP_FACTOR = 0.0625;
    private static int[] EXPECTED_CO2_BOUND = new int[]{0, 3000};

    public static byte[] decode(byte[] rawData) {
        assert (rawData.length == 8);
        byte[] res = Arrays.copyOf(rawData, 8);
        swap(res);
        res = shift(res);
        decodeWithMagicWord(res);
        return res;
    }

    static void byteSwap(byte[] data, int index1, int index2) {
        byte tmp = data[index2];
        data[index2] = data[index1];
        data[index1] = tmp;
    }

    /**
     * swap in place
     */
    private static void swap(byte[] data) {
        byteSwap(data, 0, 2);
        byteSwap(data, 1, 4);
        byteSwap(data, 3, 7);
        byteSwap(data, 5, 6);
    }


    private static byte byteShift(byte[] data, int index1, int index2) {
        return (byte) (((data[index1] << 5) & 0xe0) | ((data[index2] >> 3) & 0x1f));
    }

    /**
     * shift
     */
    private static byte[] shift(byte[] data) {
        byte[] res = new byte[8];
        res[7] = byteShift(data, 6, 7);
        res[6] = byteShift(data, 5, 6);
        res[5] = byteShift(data, 4, 5);
        res[4] = byteShift(data, 3, 4);
        res[3] = byteShift(data, 2, 3);
        res[2] = byteShift(data, 1, 2);
        res[1] = byteShift(data, 0, 1);
        res[0] = byteShift(data, 7, 0);
        return res;
    }

    /**
     * decode in place
     */
    private static void decodeWithMagicWord(byte[] data) {
        for (int i = 0; i < data.length; i++) {
            data[i] = (byte) (data[i] - MAGIC_WORD_SHIFTED[i]);
        }
    }

    public static boolean checkCRC(byte[] data) {
        if (data.length < 5) {
            return false;
        } else {
            return (data[4] == 0x0d) && ((byte)(data[0] + data[1] + data[2]) == data[3]);
        }
    }

    private static int bytesToLong(byte b1, byte b2) {
        int r1 = (b1 + 256) % 256;
        int r2 = (b2 + 256) % 256;
        return (r1 << 8) + r2;
    }

    public static Optional<MetricMessage> parseValue(byte[] data) {
        if (data.length < 5) {
            throw new IllegalArgumentException("wrong data length ${data.length} but should be >= 5");
        } else {
            byte b1 = data[1];
            byte b2 = data[2];
            long tsMillis = Instant.now().toEpochMilli();
            switch (data[0]) {
                case CO2_CODE:
                    int ppm = bytesToLong(b1, b2);
                    // according to ZG01C spec expected value in range of 0..3000
                    // device may send higher values, but them not precise
                    if (ppm < EXPECTED_CO2_BOUND[0]) {
                        return Optional.of(new MetricMessage(MetricMessage.MessageType.ERROR, -1, tsMillis));
                    } else {
                        boolean high = ppm > EXPECTED_CO2_BOUND[1];
                        if (high) {
                            ppm = EXPECTED_CO2_BOUND[1];
                        }
                        return Optional.of(new MetricMessage(MetricMessage.MessageType.CO2,  ppm, tsMillis));
                    }
                case TEMP_CODE:
                    return Optional.of(new MetricMessage(MetricMessage.MessageType.TEMPERATURE,  decodeTemperature(bytesToLong(b1, b2)), tsMillis));
                default:
                    return Optional.of(new MetricMessage(MetricMessage.MessageType.ERROR, data[0] & 0xff, tsMillis));
            }
        }
    }

    private static double decodeTemperature(int value) {
        return value * TEMP_FACTOR - ABS_ZERO_TEMP;
    }
}
