package ru.yandex.solomon.co2;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.asynchttpclient.AsyncHttpClient;

import ru.yandex.inside.solomon.SolomonStand;
import ru.yandex.inside.solomon.pusher.SolomonPusher;
import ru.yandex.misc.asyncHttpClient2.AsyncHttpClientUtils;
import ru.yandex.misc.monica.solomon.sensors.PushSensorsData;
import ru.yandex.misc.monica.solomon.sensors.Sensor;

/**
 * @author Maksim Leonov (nohttp@)
 */
public class Pusher {
    public static final long PUSH_INTERVAL_MILLIS = 15_000;

    private final String project;
    private final String cluster;
    private final String service;

    private final Map<String, String> extraLabels;

    private final Thread pusherThread;
    private final Queue<MetricMessage> messages;

    private final SolomonPusher pusher;

    public Pusher(CmdlineConfig config, ConcurrentLinkedQueue<MetricMessage> messages) {
        this.messages = messages;

        AsyncHttpClient httpClient = AsyncHttpClientUtils.newAsyncHttpClient("Solomon-pusher");
        pusher = SolomonPusher.ownHttpClient(httpClient)
            .pushTo(SolomonStand.PRODUCTION)
            .oauthToken(readFile(Path.of(config.token)))
            .build();

        project = config.project;
        cluster = config.cluster;
        service = config.service;
        extraLabels = Arrays.stream(config.labels.getOrElse("").split("&"))
                .filter(label -> !label.isEmpty())
                .map(label -> label.split("="))
                .collect(Collectors.toMap(kv -> kv[0], kv -> kv[1]));

        pusherThread = new Thread(this::pushLoop, "Pusher");
        pusherThread.setDaemon(true);
        pusherThread.start();
    }

    private static String readFile(Path path) {
        try {
            return Files.readString(path).replaceAll("[\\n\\r]", "");
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private void pushLoop() {
        while (!Thread.interrupted()) {
            try {
                if (push()) {
                    Thread.sleep(PUSH_INTERVAL_MILLIS);
                } else {
                    Thread.sleep(500);
                }
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
        closePushers();
    }

    private void closePushers() {
        pusher.close();
    }

    private boolean push() {
        MetricMessage tempMessage = null;
        MetricMessage coMessage = null;
        while (!messages.isEmpty()) {
            MetricMessage message = messages.poll();
            switch (message.messageType) {
                case CO2:
                    if (message.value > 5) {
                        coMessage = message;
                    } else {
                        // otherwise it is in init stage counting down from 5 to zero
                    }
                    break;
                case TEMPERATURE:
                    tempMessage = message;
                    break;
            }
        }

        if (tempMessage != null || coMessage != null) {
            long tsSec = roundTs(System.currentTimeMillis());
            ArrayList<Sensor> metrics = new ArrayList<>(2);
            if (tempMessage != null) {
                metrics.add(metric("temperature", tsSec, tempMessage.value));
            }
            if (coMessage != null) {
                metrics.add(metric("CO2", tsSec, coMessage.value));
            }

            return send(new PushSensorsData(project, cluster, service, extraLabels, metrics));
        } else {
            return false;
        }
    }

    private static long roundTs(long tsMillis) {
        return TimeUnit.MILLISECONDS.toSeconds(tsMillis - (tsMillis % PUSH_INTERVAL_MILLIS));
    }

    private Sensor metric(String name, long tsSec, double value) {
        return new Sensor(Map.of("sensor", name), value).overrideTs(tsSec);
    }

    private boolean send(PushSensorsData data) {
        try {
            pusher.push(data).get();
            return true;
        } catch (ExecutionException _ignore) {
            return false;
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            return false;
        }
    }

    public void stop() {
        pusherThread.interrupt();
    }
}
