package ru.yandex.solomon.experiments.alexlovkov;

import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

import com.google.common.base.Throwables;

import ru.yandex.kikimr.client.KikimrAsyncRetry;
import ru.yandex.market.graphouse.search.dao.MetricsDao;
import ru.yandex.misc.thread.ThreadUtils;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.util.ExceptionUtils;


/**
 * @author alexlovkov
 */
public class DeleteMetricsFromGraphite {

    private static final int BATCH_SIZE = 1000;

    private static final int parallelism = 16;
    private static final ExecutorService threadPool = Executors.newFixedThreadPool(parallelism);
    private static final Semaphore processSemaphore = new Semaphore(parallelism);
    private static final AtomicLong deletedMetrics = new AtomicLong();
    private static final long START = System.currentTimeMillis();
    private static final AtomicInteger inFlight = new AtomicInteger();
    private static final CompletableFuture<Void> doneFuture = new CompletableFuture<>();

    public static void main(String[] args) {
        if (args.length < 2) {
            System.err.println("require path to the file with metrics & cluster: GR/PS/BS");
            System.exit(1);
        }

        new Thread(() -> {
            for (; ; ) {
                System.out.println("semaphore:" + processSemaphore.availablePermits());
                ThreadUtils.sleep(30 * 1000);
            }
        }).start();

        String filename = args[0];
        String clusterName = args[1];

        inFlight.incrementAndGet();
        try (var reader = Files.newBufferedReader(Path.of(filename), StandardCharsets.UTF_8);
             var ydb = YdbHelper.createYdbClient(SolomonCluster.PROD_FRONT))
        {
            var dao = DownloadMetricsFromDb.getMetricsDaoKikimr(clusterName, ydb);
            String line;
            List<String> metrics = new ArrayList<>();
            while ((line = reader.readLine()) != null) {
                metrics.add(line);
                if (metrics.size() >= BATCH_SIZE) {
                    deleteMetrics(dao, metrics);
                    metrics = new ArrayList<>();
                }
            }
            deleteMetrics(dao, metrics);
        } catch (Throwable e) {
            System.out.println(Throwables.getStackTraceAsString(e));
            System.out.println("FAILED");
            System.exit(1);
        }

        completeOperation();
        doneFuture.join();
        System.out.println("FINISHED");
    }

    private static void completeOperation() {
        if (inFlight.decrementAndGet() == 0) {
            doneFuture.complete(null);
        }
    }

    private static void deleteMetrics(
        MetricsDao dao,
        List<String> metrics)
    {
        try {
            processSemaphore.acquire();
        } catch (InterruptedException e) {
            ExceptionUtils.uncaughtException(e);
        }

        threadPool.execute(() -> {
            try {
                long start = System.currentTimeMillis();
                KikimrAsyncRetry.withRetries(KikimrAsyncRetry.RetryConfig.maxRetriesAndTime(1000, 5 * 1000),
                    () -> dao.remove(metrics)).join();
                System.out.println(
                    "deleted:" + metrics.size() + " metrics for iteration for:" + (System.currentTimeMillis() - start));
                System.out.println("deleted " + deletedMetrics.addAndGet(metrics.size()) + " for " +
                    (System.currentTimeMillis() - START) + " ms");
            } catch (Throwable t) {
                ExceptionUtils.uncaughtException(t);
            } finally {
                processSemaphore.release();
                completeOperation();
            }
        });
    }
}
