package ru.yandex.solomon.experiments.alexlovkov;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.concurrent.atomic.AtomicInteger;

import com.google.common.base.Throwables;
import com.google.common.util.concurrent.MoreExecutors;

import ru.yandex.market.graphouse.search.MetricDbRow;
import ru.yandex.market.graphouse.search.dao.MetricsDao;
import ru.yandex.market.graphouse.search.dao.ydb.YdbMetricsDao;
import ru.yandex.solomon.config.protobuf.Time;
import ru.yandex.solomon.config.protobuf.TimeUnit;
import ru.yandex.solomon.config.protobuf.graphite.storage.TKikimrConfig;
import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;


/**
 * @author alexlovkov
 */
public class DownloadMetricsFromDb {

    public static void main(String[] args) throws IOException {
        saveMetricsFromDbToFile(args);
    }

    private static void saveMetricsFromDbToFile(String[] args) {
        if (args.length != 2) {
            System.err.println("specify the file and the cluster BS/GR/PS");
            System.exit(1);
        }

        String file = args[0];
        String cluster = args[1];

        AtomicInteger count = new AtomicInteger(0);
        try (var bw = Files.newBufferedWriter(Paths.get(file));
             var ydb = YdbHelper.createYdbClient(SolomonCluster.PROD_FRONT))
        {
            var dao = getMetricsDaoKikimr(cluster, ydb);
            dao.loadAllMetrics(list -> {
                try {
                    for (MetricDbRow s : list) {
                        String str = s.shardId() + "|" + s.localId() + "|" + s.metricName() + "\n";
                        bw.write(str);
                    }
                    int cnt = count.addAndGet(list.size());
                    if (cnt % 100_000 == 0) {
                        System.out.println("loaded " + cnt + " metrics");
                    }
                } catch (Exception e) {
                    System.out.println(Throwables.getStackTraceAsString(e));
                }
            }).join();
        } catch (Throwable e) {
            System.out.println(Throwables.getStackTraceAsString(e));
            System.out.println("FAILED");
            System.exit(1);
        }
        System.out.println("FINISHED");
        System.exit(0);
    }

    public static MetricsDao getMetricsDaoKikimr(String cluster, YdbClient client) {
        TKikimrConfig.Builder kikimrConfigBuilder = TKikimrConfig.newBuilder();
        Time.Builder timeBuilder = Time.newBuilder();
        timeBuilder.setUnit(TimeUnit.SECONDS);
        timeBuilder.setValue(10);
        kikimrConfigBuilder.setLastUpdateTtl(timeBuilder);
        kikimrConfigBuilder.setSchemaRoot("/Kfront/MegaGraphite/" + cluster);

        Time ttl = Time.newBuilder().setValue(10).setUnit(TimeUnit.SECONDS).build();
        kikimrConfigBuilder.setLastUpdateTtl(ttl);
        kikimrConfigBuilder.setMaxDeletionInFlight(5);
        kikimrConfigBuilder.setBatchSize(5);
        kikimrConfigBuilder.setMaxSavingInFlight(25);

        return new YdbMetricsDao(kikimrConfigBuilder.build(), client.table, MoreExecutors.directExecutor());
    }
}
