package ru.yandex.solomon.experiments.alexlovkov;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author alexlovkov
 */
public class Trie {

    private static final double PERCENT_OF_CHILDREN = 0.8;
    private static final int MIN_CHILDREN_COUNT = 50000;

    private TrieNode root;
    private TrieNode current;

    public Trie() {
        this.root = new TrieNode("ROOT", null);
        this.current = root;
    }

    public void insert(String[] strings) {
        TrieNode current = root;
        for (String s : strings) {
            TrieNode cur = current;
            current = current.getChildren().computeIfAbsent(s, c -> new TrieNode(s, cur));
            current.incrementCount();
        }
        current.setLeaf(true);
    }

    public boolean find(String[] strings) {
        TrieNode current = root;
        for (String s : strings) {
            TrieNode node = current.getChildren().get(s);
            if (node == null) {
                return false;
            }
            current = node;
        }
        return current.isLeaf();
    }

    public void analyze() {
        analyze(root, "");
    }

    private int analyze(TrieNode current, String path) {
        String p;
        if (path.isEmpty()) {
            p = path;
        } else {
            p = path + '.';
        }
        String newPath = p;
        int childrenCount = current.getChildren().values().stream().map(
            child -> analyze(child, newPath + child.getContent()))
            .mapToInt(Integer::intValue).sum();

        int count = childrenCount + 1;

        int children = current.getChildren().size();
        if (children > count * PERCENT_OF_CHILDREN && count > MIN_CHILDREN_COUNT) {
            System.out.println(path + " children:" + children + " count:" + count);
        }
        return count;
    }

    private class TrieNode {

        private final Map<String, TrieNode> children;
        private final String content;
        private boolean leaf;
        private final TrieNode parent;
        private long count;

        public TrieNode(String content, TrieNode parent) {
            this.children = new HashMap<>();
            this.leaf = false;
            this.content = content;
            this.parent = parent;
            this.count = 0;
        }

        public Map<String, TrieNode> getChildren() {
            return children;
        }

        public String getContent() {
            return content;
        }

        public boolean isLeaf() {
            return leaf;
        }

        public void setLeaf(boolean leaf) {
            this.leaf = leaf;
        }

        public TrieNode getParent() {
            return parent;
        }

        public long getCount() {
            return count;
        }

        public void incrementCount() {
            this.count++;
        }
    }


    public List<Node> getChildren() {
        return current.getChildren().values().stream()
            .map(e -> new Node(e.getContent(), e.getCount()))
            .collect(Collectors.toList());
    }

    public void goToChild(String child) {
        current = current.getChildren().get(child);
        if (current == null) {
            throw new IllegalArgumentException("child:" + child);
        }
    }

    public void goToParent() {
        if (current.getParent() == null) {
            return;
        }
        current = current.getParent();
    }

}
