package ru.yandex.solomon.experiments.alextrushkin;

import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.netty.handler.codec.http.HttpHeaderNames;

import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.solomon.alert.dao.AlertTemplateDao;
import ru.yandex.solomon.alert.dao.AlertTemplateLastVersionDao;
import ru.yandex.solomon.alert.dao.AlertsDao;
import ru.yandex.solomon.alert.dao.ydb.YdbAlertTemplateDao;
import ru.yandex.solomon.alert.dao.ydb.YdbAlertTemplateLastVersionDao;
import ru.yandex.solomon.alert.dao.ydb.YdbSchemaVersion;
import ru.yandex.solomon.alert.dao.ydb.entity.YdbAlertsDao;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.AlertState;
import ru.yandex.solomon.alert.domain.template.AlertFromTemplatePersistent;
import ru.yandex.solomon.alert.domain.template.AlertParameter;
import ru.yandex.solomon.alert.template.domain.AlertTemplate;
import ru.yandex.solomon.alert.template.domain.AlertTemplateId;
import ru.yandex.solomon.alert.template.domain.AlertTemplateParameter;
import ru.yandex.solomon.auth.exceptions.AuthorizationException;
import ru.yandex.solomon.idempotency.IdempotentOperation;
import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;

/**
 * @author Alexey Trushkin
 */
public class AlertsForProviderStatistics implements AutoCloseable {

    private final YdbClient ydb;
    private final String ytTable;
    private final String ytAddress;
    private final String project;
    private final String serviceProvider;
    private final AlertsDao alertsDao;
    private final AlertTemplateDao alertTemplateDao;
    private final AlertTemplateLastVersionDao alertTemplateLastVersionDao;

    public AlertsForProviderStatistics(SolomonCluster cluster, String ytTable, String yt, String project, String serviceProvider) {
        this.ydb = YdbHelper.createYdbClient(cluster);
        this.ytTable = ytTable;
        this.ytAddress = yt;
        this.project = project;
        this.serviceProvider = serviceProvider;
        var mapper = new ObjectMapper();
        alertsDao = new YdbAlertsDao(cluster.kikimrRootPath(), ydb.table, ydb.scheme, YdbSchemaVersion.CURRENT, mapper);
        alertTemplateDao = new YdbAlertTemplateDao(cluster.kikimrRootPath(), ydb.table, ydb.scheme, YdbSchemaVersion.CURRENT, mapper);
        alertTemplateLastVersionDao = new YdbAlertTemplateLastVersionDao(cluster.kikimrRootPath(), ydb.table, ydb.scheme, YdbSchemaVersion.CURRENT);
    }

    private void createAlerts() {
        Yt yt = YtUtils.http(ytAddress);
        YPath table = YPath.simple(ytTable);

        var published = alertTemplateLastVersionDao.getAll().join().stream()
                .map(version -> new AlertTemplateId(version.id(), version.templateVersionTag()))
                .collect(Collectors.toSet());
        var templates = alertTemplateDao.getAll().join().stream()
                .filter(alertTemplate -> published.contains(alertTemplate.getCompositeId()))
                .filter(alertTemplate -> alertTemplate.getServiceProviderId().equals(serviceProvider))
                .collect(Collectors.toList());


        yt.tables().read(
                table.withColumns("resource_id"),
                YTableEntryTypes.YSON,
                (entry) -> {
                    final YTreeMapNode resource_id = entry.getMap("resource_id");
                    for (var template : templates) {
                        List<AlertParameter> params = fillParams(template, resource_id);
                        List<AlertParameter> thresholds = fillThresholds(template);
                        var alert = AlertFromTemplatePersistent.newBuilder()
                                .setId(UUID.randomUUID().toString())
                                .setName(template.getName() + "(" + resource_id + ")")
                                .setProjectId(project)
                                .setState(AlertState.ACTIVE)
                                .setTemplateId(template.getId())
                                .setTemplateVersionTag(template.getTemplateVersionTag())
                                .setParameters(params)
                                .setThresholds(thresholds)
                                .build();
                        alertsDao.insert(alert, IdempotentOperation.NO_OPERATION).join();
                    }
                }
        );
    }

    private List<AlertParameter> fillThresholds(AlertTemplate template) {
        List<AlertParameter> result = new ArrayList<>();
        for (AlertTemplateParameter parameter : template.getThresholds()) {
            if (parameter.getType() == AlertTemplateParameter.ParameterValueType.DOUBLE) {
                result.add(new AlertParameter.DoubleParameterValue(((AlertTemplateParameter.DoubleParameterValue) parameter).getDefaultValue(), parameter.getName()));
            } else if (parameter.getType() == AlertTemplateParameter.ParameterValueType.INTEGER) {
                result.add(new AlertParameter.IntegerParameterValue(((AlertTemplateParameter.IntegerParameterValue) parameter).getDefaultValue(), parameter.getName()));
            }
        }
        return result;
    }

    private List<AlertParameter> fillParams(AlertTemplate template, YTreeMapNode resource_id) {
        List<AlertParameter> result = new ArrayList<>();
        for (AlertTemplateParameter parameter : template.getParameters()) {
            result.add(new AlertParameter.TextParameterValue(resource_id.getString(parameter.getName()), parameter.getName()));
        }
        return result;
    }

    public static void main(String... args) {
        var cluster = SolomonCluster.PROD_KFRONT;
        var serviceProvider = "monitoring";
        var project = "alextrushkin_" + serviceProvider;
        var table = "//home/solomon/service_provider_alerts/service_provider_exports/" + serviceProvider;
        var yt = "hahn.yt.yandex.net";
        try (var cli = new AlertsForProviderStatistics(cluster, table, yt, project, serviceProvider)) {
            //  cli.createAlerts();
            cli.stats();
            // cli.deleteAlerts();
        } catch (Throwable t) {
            t.printStackTrace();
            System.exit(1);
        }
        System.exit(0);
    }

    private void stats() {
        var httpClient = HttpClient.newBuilder()
                .version(HttpClient.Version.HTTP_1_1)
                .followRedirects(HttpClient.Redirect.NEVER)
                .connectTimeout(Duration.ofSeconds(120))
                .executor(Executors.newSingleThreadExecutor())
                .build();
        ObjectMapper mapper = new ObjectMapper();
        var token = System.getProperty("OAUTH_TOKEN");
        if (token == null || token.isEmpty()) {
            throw new AuthorizationException("no token");
        }
        var pageToken = "0";
        var ok = 0;
        var alarm = 0;
        var warn = 0;
        var noData = 0;
        var error = 0;
        var count = 0;
        do {
            String alertPath = "https://solomon.yandex-team.ru/api/v2/projects/" + project + "/alerts?pageSize=900&pageToken=" + pageToken;
            var request = HttpRequest.newBuilder(URI.create(alertPath))
                    .GET()
                    .header(HttpHeaderNames.AUTHORIZATION.toString(), "OAuth " + token)
                    .build();
            HttpResponse<String> alertsResponse = httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString()).join();
            try {
                Map<String, Object> response = mapper.readValue(alertsResponse.body(), new TypeReference<Map<String, Object>>() {
                });

                pageToken = response.getOrDefault("nextPageToken", "").toString();
                List<Map<String, Object>> alerts = (List<Map<String, Object>>) response.getOrDefault("items", List.of());
                for (Map<String, Object> alert : alerts) {
                    Map<String, Object> stats = (Map<String, Object>) alert.getOrDefault("evaluationStats", Map.of());
                    for (Map.Entry<String, Object> statEntry : stats.entrySet()) {
                        if (statEntry.getKey().equals("ok")) {
                            ok += (int) statEntry.getValue();
                            count += (int) statEntry.getValue();
                        } else if (statEntry.getKey().equals("noData")) {
                            noData += (int) statEntry.getValue();
                            count += (int) statEntry.getValue();
                        } else if (statEntry.getKey().equals("error")) {
                            error += (int) statEntry.getValue();
                            count += (int) statEntry.getValue();
                        } else if (statEntry.getKey().equals("alarm")) {
                            alarm += (int) statEntry.getValue();
                            count += (int) statEntry.getValue();
                        } else if (statEntry.getKey().equals("warn")) {
                            warn += (int) statEntry.getValue();
                            count += (int) statEntry.getValue();
                        }
                    }
                }
            } catch (Throwable t) {
                throw new RuntimeException(t);
            }
        } while (!pageToken.equals(""));

        System.out.println("Project: https://solomon.yandex-team.ru/admin/projects/" + project + "/alerts?templateServiceProviderId=ALL");
        System.out.println("ServiceProvider: " + serviceProvider);
        System.out.println("Alerts(with subalerts) count: " + count);
        System.out.println("ok: " + ok);
        System.out.println("warn: " + warn);
        System.out.println("alarm: " + alarm);
        System.out.println("error: " + error);
        System.out.println("noData: " + noData);
        System.out.println();
        System.out.println("alarm %: " + ((double) alarm / count * 100));
        System.out.println("warn %: " + ((double) warn / count * 100));
        assert count != ok + warn + alarm + error + noData;
    }

    private void deleteAlerts() {
        final List<Alert> alerts = alertsDao.findAll(project).join();
        for (Alert alert : alerts) {
            alertsDao.deleteById(project, alert.getId(), IdempotentOperation.NO_OPERATION).join();
        }
    }

    @Override
    public void close() {
        ydb.close();
    }
}
