package ru.yandex.solomon.experiments.alextrushkin;

import java.io.File;
import java.util.Map;

import javax.net.ssl.SSLException;

import io.grpc.ManagedChannel;
import io.grpc.netty.GrpcSslContexts;
import io.grpc.netty.NegotiationType;
import io.grpc.netty.NettyChannelBuilder;

import ru.yandex.monitoring.api.v3.CreateDashboardRequest;
import ru.yandex.monitoring.api.v3.Dashboard;
import ru.yandex.monitoring.api.v3.DashboardServiceGrpc;
import ru.yandex.monitoring.api.v3.DeleteDashboardRequest;
import ru.yandex.monitoring.api.v3.GetDashboardRequest;
import ru.yandex.monitoring.api.v3.ListDashboardOperationsRequest;
import ru.yandex.monitoring.api.v3.ListDashboardsRequest;
import ru.yandex.monitoring.api.v3.UpdateDashboardRequest;
import ru.yandex.monitoring.api.v3.Widget;
import ru.yandex.solomon.util.Proto;

/**
 * @author Alexey Trushkin
 */
public class DashboardV3Client {

    public static void main(String[] args) throws Exception {
        AuthCallCredentials credentials = AuthCallCredentials.asUser("alextrushkin");
        var channel = localChannel();
        var client = DashboardServiceGrpc.newBlockingStub(channel)
                .withCallCredentials(credentials);

        // List dashboards by name filter
        var listRequest = ListDashboardsRequest.newBuilder()
                .setProjectId("solomon")
                .setFilter("name=\"GRPC\"")
                .build();
        var response = client.list(listRequest);
        Dashboard myDashboard = null;
        for (Dashboard dashboard : response.getDashboardsList()) {
            if (dashboard.getId().equals("monkn6m5ui3ja9l5lvlp")) {
                myDashboard = dashboard;
            }
        }

        // Create new dashboard
        var baseWidget = myDashboard.getWidgetsList().get(0);
        var createOperation = client.create(CreateDashboardRequest.newBuilder()
                .setProjectId(myDashboard.getProjectId())
                .setName("my-dashboard")
                .setDescription(myDashboard.getDescription())
                .putAllLabels(Map.of("label1", "value2"))
                .setTitle("Title")
                .addWidgets(Widget.newBuilder()
                        .setChart(baseWidget.getChart())
                        .setPosition(baseWidget.getPosition())
                        .build())
                .setParametrization(myDashboard.getParametrization())
                .build());
        if (createOperation.hasError()) {
            throw new IllegalArgumentException(createOperation.getError().getMessage());
        }
        var createdDashboard = Proto.unpack(createOperation.getResponse(), Dashboard.class);
        System.out.println("createdDashboard: " + createdDashboard.getTitle());

        // Update dashboard by id
        var updateOperation = client.update(UpdateDashboardRequest.newBuilder()
                .setDashboardId(createdDashboard.getId())
                .setName(createdDashboard.getName())
                .setDescription(createdDashboard.getDescription())
                .putAllLabels(Map.of("l1", "v22"))
                .setTitle(createdDashboard.getName())
                .addAllWidgets(createdDashboard.getWidgetsList())
                .setParametrization(createdDashboard.getParametrization())
                .setEtag(createdDashboard.getEtag())
                .build());
        if (updateOperation.hasError()) {
            throw new IllegalArgumentException(updateOperation.getError().getMessage());
        }
        var updatedDashboard = Proto.unpack(updateOperation.getResponse(), Dashboard.class);
        System.out.println("updatedDashboard: " + updatedDashboard.getTitle());

        // Get dashboard by id
        var dashboard = client.get(GetDashboardRequest.newBuilder()
                .setDashboardId(createdDashboard.getId())
                .build());
        System.out.println("dashboard: " + dashboard);

        // Get dashboard operations
        var operations = client.listOperations(ListDashboardOperationsRequest.newBuilder()
                .setDashboardId(dashboard.getId())
                .build());
        System.out.println("operations: " + operations);

        // Delete dashboard by id
        var deleteOperation = client.delete(DeleteDashboardRequest.newBuilder()
                .setDashboardId(dashboard.getId())
                .setEtag(dashboard.getEtag())
                .build());
        if (deleteOperation.hasError()) {
            throw new IllegalArgumentException(deleteOperation.getError().getMessage());
        }
        System.out.println("deleteOperation: " + deleteOperation);
    }

    private static ManagedChannel localChannel() {
        return NettyChannelBuilder.forTarget("localhost:5740")
                .negotiationType(NegotiationType.PLAINTEXT)
                .build();
    }


    private static ManagedChannel remoteChannel(String address) throws SSLException {
        if (address.isBlank()) {
            address = "monitoring.private-api.cloud-preprod.yandex.net:443";
        }
        var sslContext = GrpcSslContexts.forClient().trustManager(new File("/Users/alextrushkin/arcadia/certs/cacert.pem")).build();
        return NettyChannelBuilder.forTarget(address)
                .negotiationType(NegotiationType.TLS)
                .sslContext(sslContext)
                .build();
    }
}
