package ru.yandex.solomon.experiments.alextrushkin;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ForkJoinPool;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectMapper;

import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.solomon.alert.dao.AlertTemplateDao;
import ru.yandex.solomon.alert.dao.AlertsDao;
import ru.yandex.solomon.alert.dao.ydb.YdbAlertTemplateDao;
import ru.yandex.solomon.alert.dao.ydb.YdbSchemaVersion;
import ru.yandex.solomon.alert.dao.ydb.entity.YdbAlertsDao;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.template.AlertFromTemplatePersistent;
import ru.yandex.solomon.alert.domain.template.AlertParameter;
import ru.yandex.solomon.alert.template.domain.AlertTemplate;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbProjectsDao;
import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;

/**
 * @author Alexey Trushkin
 */
public class ExportAlertFromTemplates implements AutoCloseable {

    private final YdbClient ydb;
    private final String ytTable;
    private final String ytAddress;
    private final boolean skipYasm;
    private final AlertsDao alertsDao;
    private final AlertTemplateDao alertTemplateDao;
    private final ProjectsDao projectsDao;

    public ExportAlertFromTemplates(SolomonCluster cluster, String ytTable, String yt, boolean skipYasm) {
        this.ydb = YdbHelper.createYdbClient(cluster);
        this.ytTable = ytTable;
        this.ytAddress = yt;
        this.skipYasm = skipYasm;
        var mapper = new ObjectMapper();
        alertsDao = new YdbAlertsDao(cluster.kikimrRootPath(), ydb.table, ydb.scheme, YdbSchemaVersion.CURRENT, mapper);
        alertTemplateDao = new YdbAlertTemplateDao(cluster.kikimrRootPath(), ydb.table, ydb.scheme, YdbSchemaVersion.CURRENT, mapper);
        projectsDao = new YdbProjectsDao(ydb.table, cluster.kikimrRootPath() + "/Config/V2/Project", mapper, ForkJoinPool.commonPool());
    }

    private void exportYtTable() {
        Yt yt = YtUtils.http(ytAddress);
        YPath table = YPath.simple(ytTable);

        var projects = projectsDao.findAllNames().join();
        var templateToSp = alertTemplateDao.getAll().join().stream()
                .collect(Collectors.toMap(AlertTemplate::getId, AlertTemplate::getServiceProviderId, (s, s2) -> s));

        List<YTreeMapNode> entries = new ArrayList<>();
        for (var project : projects) {
            if (skipYasm && project.getId().startsWith("yasm_")) {
                continue;
            }
            int alertCount = 0;
            var alerts = alertsDao.findAll(project.getId()).join();
            for (Alert alert : alerts) {
                if (alert instanceof AlertFromTemplatePersistent alertFromTemplate) {
                    Map<String, String> resourceId = new HashMap<>();
                    for (AlertParameter parameter : alertFromTemplate.getParameters()) {
                        if (parameter instanceof AlertParameter.TextListParameterValue v) {
                            resourceId.put(parameter.getName(), String.join("|", v.getValue()));
                        } else if (parameter instanceof AlertParameter.TextParameterValue v) {
                            resourceId.put(parameter.getName(), v.getValue());
                        } else if (parameter instanceof AlertParameter.LabelListParameterValue v) {
                            resourceId.put(parameter.getName(), String.join("|", v.getValues()));
                        } else if (parameter instanceof AlertParameter.IntegerParameterValue v) {
                            resourceId.put(parameter.getName(), v.getValue() + "");
                        } else if (parameter instanceof AlertParameter.DoubleParameterValue v) {
                            resourceId.put(parameter.getName(), v.getValue() + "");
                        }
                    }
                    var serviceProvider = templateToSp.get(alertFromTemplate.getTemplateId());
                    entries.add(createEntry(serviceProvider,
                            resourceId,
                            alertFromTemplate.getTemplateId(),
                            project.getId(),
                            project.getAbcService()));
                    alertCount++;
                }
            }
            System.out.println("Project '" + project.getId() + "' collected alerts: " + alertCount);
        }

        yt.tables().write(table, YTableEntryTypes.YSON, entries);
        System.out.println("Exported alerts: " + entries.size());

    }

    private YTreeMapNode createEntry(
            String serviceProvider,
            Map<String, String> resource,
            String templateId,
            String projectId,
            String abcSlug)
    {
        return YTree.mapBuilder()
                .key("service_provider_id").value(serviceProvider)
                .key("resource_id").value(resource)
                .key("template_id").value(templateId)
                .key("project_id").value(projectId)
                .key("abc_slug").value(abcSlug)
                .buildMap();
    }

    public static void main(String... args) {
        var skipYasm = true;
        var cluster = SolomonCluster.PROD_KFRONT;
        var table = "//home/solomon/service_provider_alerts/alerts/alerts_all";
        var yt = "hahn.yt.yandex.net";
        System.out.println("Start export template alerts from " + cluster + " to " + yt + table);
        try (var cli = new ExportAlertFromTemplates(cluster, table, yt, skipYasm)) {
            cli.exportYtTable();
        } catch (Throwable t) {
            t.printStackTrace();
            System.exit(1);
        }
        System.out.println("Export finished");
        System.exit(0);
    }

    @Override
    public void close() {
        ydb.close();
    }
}
