package ru.yandex.solomon.experiments.alextrushkin;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ForkJoinPool;

import com.fasterxml.jackson.databind.ObjectMapper;

import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbProjectsDao;
import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;

/**
 * @author Alexey Trushkin
 */
public class ExportAlertFromTemplatesForMonitoring implements AutoCloseable {

    private final YdbClient ydb;
    private final String ytTable;
    private final String ytAddress;
    private final boolean skipYasm;
    private final ProjectsDao projectsDao;

    public ExportAlertFromTemplatesForMonitoring(SolomonCluster cluster, String ytTable, String yt, boolean skipYasm) {
        this.ydb = YdbHelper.createYdbClient(cluster);
        this.ytTable = ytTable;
        this.ytAddress = yt;
        this.skipYasm = skipYasm;
        var mapper = new ObjectMapper();
        projectsDao = new YdbProjectsDao(ydb.table, cluster.kikimrRootPath() + "/Config/V2/Project", mapper, ForkJoinPool.commonPool());
    }

    private void exportYtTable() {
        Yt yt = YtUtils.http(ytAddress);
        YPath table = YPath.simple(ytTable);

        var projects = projectsDao.findAllNames().join();
        List<YTreeMapNode> entries = new ArrayList<>();
        for (var project : projects) {
            if (skipYasm && project.getId().startsWith("yasm_")) {
                continue;
            }
            entries.add(createEntry(Map.of("projectId", project.getId()),
                    project.getOwner(),
                    project.getAbcService()));
        }

        yt.tables().write(table, YTableEntryTypes.YSON, entries);
        System.out.println("Exported projects: " + entries.size());

    }

    private YTreeMapNode createEntry(
            Map<String, String> resource,
            String responsible,
            String abcSlug) {
        return YTree.mapBuilder()
                .key("service_provider_id").value("monitoring")
                .key("resource_id").value(resource)
                .key("responsible").value(responsible)
                .key("resource_type").value("project")
                .key("environment").value("PRODUCTION")
                .key("abc_slug").value(abcSlug)
                .key("abc_id").value(-1)
                .buildMap();
    }

    public static void main(String... args) {
        var skipYasm = true;
        var cluster = SolomonCluster.PROD_KFRONT;
        var table = "//home/solomon/service_provider_alerts/service_provider_exports/monitoring";
        var yt = "hahn.yt.yandex.net";
        System.out.println("Start export resources from " + cluster + " to " + yt + table);
        try (var cli = new ExportAlertFromTemplatesForMonitoring(cluster, table, yt, skipYasm)) {
            cli.exportYtTable();
        } catch (Throwable t) {
            t.printStackTrace();
            System.exit(1);
        }
        System.out.println("Export finished");
        System.exit(0);
    }

    @Override
    public void close() {
        ydb.close();
    }
}
