package ru.yandex.solomon.experiments.alextrushkin;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jetbrains.annotations.NotNull;

import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;

/**
 * @author Alexey Trushkin
 */
public class ExportMdbResources implements AutoCloseable {

    private final String ytTablePrefix;
    private final String ytAddress;
    private final Set<String> skippedStatuses = new HashSet<>(Arrays.asList("DELETING",
            "DELETE-ERROR",
            "DELETED",
            "METADATA-DELETING",
            "METADATA-DELETE-ERROR",
            "METADATA-DELETED",
            "PURGING",
            "PURGE-ERROR",
            "PURGED"));

    public ExportMdbResources(String ytTablePrefix, String yt) {
        this.ytTablePrefix = ytTablePrefix;
        this.ytAddress = yt;
    }

    private void exportYtTable() {
        Yt yt = YtUtils.http(ytAddress);
        YPath cloudsTable = YPath.simple("//home/cloud-dwh/data/prod_internal/ods/mdb/clouds");
        Map<Integer, Cloud> cloudMap = new HashMap<>();
        yt.tables().read(
                cloudsTable.withColumns("cloud_ext_id", "cloud_id"),
                YTableEntryTypes.YSON,
                (entry) -> {
                    var cloud_ext_id = entry.getString("cloud_ext_id");
                    var cloud_id = entry.getInt("cloud_id");
                    cloudMap.put(cloud_id, new Cloud(cloud_id, cloud_ext_id));
                }
        );

        YPath foldersTable = YPath.simple("//home/cloud-dwh/data/prod_internal/ods/mdb/folders");
        Map<Integer, Folder> foldersMap = new HashMap<>();
        yt.tables().read(
                foldersTable.withColumns("folder_ext_id", "cloud_id", "folder_id"),
                YTableEntryTypes.YSON,
                (entry) -> {
                    var folder_id = entry.getInt("folder_id");
                    var folder_ext_id = entry.getString("folder_ext_id");
                    var cloud_id = entry.getInt("cloud_id");
                    foldersMap.put(folder_id, new Folder(cloud_id, folder_id, folder_ext_id));
                }
        );

        YPath clustersTable = YPath.simple("//home/cloud-dwh/data/prod_internal/ods/mdb/clusters");
        Map<String, Cluster> clustersMap = new HashMap<>();
        yt.tables().read(
                clustersTable.withColumns("cid", "type", "folder_id", "env", "status"),
                YTableEntryTypes.YSON,
                (entry) -> {
                    var type = entry.getString("type");
                    var cid = entry.getString("cid");
                    var folder_id = entry.getInt("folder_id");
                    var env = entry.getString("env");
                    var status = entry.getString("status");
                    clustersMap.put(cid, new Cluster(cid, folder_id, type, env, status));
                }
        );
        YPath abcTable = YPath.simple(ytTablePrefix + "cloud_to_abc");
        Map<Integer, Abc> abcMap = new HashMap<>();
        yt.tables().read(
                abcTable.withColumns("cloud_id", "cloud_ext_id", "abc_slug", "abc_id"),
                YTableEntryTypes.YSON,
                (entry) -> {
                    var abc_slug = entry.getString("abc_slug");
                    var abc_id = entry.getInt("abc_id");
                    var cloud_id = entry.getInt("cloud_id");
                    var cloud_ext_id = entry.getString("cloud_ext_id");
                    abcMap.put(cloud_id, new Abc(cloud_id, cloud_ext_id, abc_slug, abc_id));
                }
        );

        Map<String, List<YTreeMapNode>> entriesMap = new HashMap<>();
        for (var cluster : clustersMap.values()) {
            if (skippedStatuses.contains(cluster.status)) {
                continue;
            }
            String env = getEnv(cluster);
            String type = getType(cluster);
            if (type == null) {
                continue;
            }
            var list = entriesMap.computeIfAbsent(type, s -> new ArrayList<>());
            var abc = resolveAbc(abcMap, cloudMap, foldersMap, cluster);
            if (abc == null) {
                continue;
            }
            list.add(createEntry(Map.of("cluster", cluster.cid()),
                    type,
                    env,
                    abc,
                    cluster.cid()));
        }

        for (Map.Entry<String, List<YTreeMapNode>> stringListEntry : entriesMap.entrySet()) {
            YPath table = YPath.simple(ytTablePrefix + stringListEntry.getKey());
            yt.tables().write(table, YTableEntryTypes.YSON, stringListEntry.getValue());
            System.out.println("Exported " + stringListEntry.getKey() + ": " + stringListEntry.getValue().size());
        }
    }

    private String getType(Cluster cluster) {
        var type = cluster.type;
        if (type.contains("clickhouse")) {
            type = "managed-clickhouse";
        } else if (type.contains("elasticsearch")) {
            type = "managed-elasticsearch";
        } else if (type.contains("kafka")) {
            type = "managed-kafka";
        } else if (type.contains("mongodb")) {
            type = "managed-mongodb";
        } else if (type.contains("mysql")) {
            type = "managed-mysql";
        } else if (type.contains("postgresql")) {
            type = "managed-postgresql";
        } else if (type.contains("redis")) {
            type = "managed-redis";
        } else {
            System.out.println(cluster.cid + " unknown type " + type);
            return null;
        }
        return type;
    }

    @NotNull
    private String getEnv(Cluster cluster) {
        var env = cluster.env;
        if (env.equals("prod")) {
            env = "PRODUCTION";
        } else if (env.equals("qa")) {
            env = "PRESTABLE";
        } else if (!env.isEmpty()) {
            env = "PRODUCTION";
        }
        return env;
    }

    private Abc resolveAbc(Map<Integer, Abc> abc, Map<Integer, Cloud> cloudMap, Map<Integer, Folder> foldersMap, Cluster cluster) {
        if (failedIds.contains(cluster.folderId)) {
            return null;
        }
        final Folder folder = foldersMap.get(cluster.folderId);
        if (folder == null) {
            failedIds.add(cluster.folderId);
            System.out.println(cluster.cid + " no folder " + cluster.folderId + " in table");
            return null;
        }
        if (failedIds.contains(folder.cloudId)) {
            return null;
        }
        final Cloud cloud = cloudMap.get(folder.cloudId);
        if (cloud == null) {
            failedIds.add(folder.cloudId);
            System.out.println(cluster.cid + " no cloud " + folder.cloudId + " in table");
            return null;
        }
        final Abc abcRecord = abc.get(cloud.cloudId);
        if (abcRecord == null) {
            failedIds.add(folder.cloudId);
            System.out.println(cluster.cid + " no abc " + folder.cloudId + " in table");
            return null;
        }
        return abcRecord;
    }

    private Set<Integer> failedIds = new HashSet<>();

    private YTreeMapNode createEntry(
            Map<String, String> resource,
            String type,
            String env,
            Abc abc,
            String global_id) {
        return YTree.mapBuilder()
                .key("service_provider_id").value(type)
                .key("resource_id").value(resource)
                .key("responsible").value("")
                .key("global_id").value(global_id)
                .key("resource_type").value("cluster")
                .key("environment").value(env)
                .key("abc_slug").value(abc.abcSlug())
                .key("abc_id").value(abc.abcId())
                .buildMap();
    }

    public static void main(String... args) {
        var table = "//home/solomon/service_provider_alerts/service_provider_exports/";
        var yt = "hahn.yt.yandex.net";
        System.out.println("Start export resources from  MDB to " + yt + table);
        try (var cli = new ExportMdbResources(table, yt)) {
            cli.exportYtTable();
        } catch (Throwable t) {
            t.printStackTrace();
            System.exit(1);
        }
        System.out.println("Export finished");
        System.exit(0);
    }

    @Override
    public void close() {

    }

    private static record Cloud(int cloudId, String cloudExtId) {

    }

    private static record Folder(int cloudId, int folderId, String folderExtId) {

    }

    private static record Cluster(String cid, int folderId, String type, String env, String status) {

    }

    private static record Abc(int cloudId, String cloudExtId, String abcSlug, int abcId) {

    }
}
