package ru.yandex.solomon.experiments.alextrushkin;

import java.io.File;
import java.util.Map;
import java.util.UUID;

import javax.net.ssl.SSLException;

import io.grpc.ManagedChannel;
import io.grpc.netty.GrpcSslContexts;
import io.grpc.netty.NegotiationType;
import io.grpc.netty.NettyChannelBuilder;

import ru.yandex.monitoring.api.v3.project.manager.AccessServiceGrpc;
import ru.yandex.monitoring.api.v3.project.manager.AuthorizeRequest;
import ru.yandex.monitoring.api.v3.project.manager.AvailableResourcesRequest;
import ru.yandex.monitoring.api.v3.project.manager.CreateProjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.DeleteProjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.GetProjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.ListProjectOperationsRequest;
import ru.yandex.monitoring.api.v3.project.manager.ListProjectsRequest;
import ru.yandex.monitoring.api.v3.project.manager.Project;
import ru.yandex.monitoring.api.v3.project.manager.ProjectServiceGrpc;
import ru.yandex.monitoring.api.v3.project.manager.Resource;
import ru.yandex.monitoring.api.v3.project.manager.ResourcePath;
import ru.yandex.monitoring.api.v3.project.manager.Role;
import ru.yandex.monitoring.api.v3.project.manager.Subject;
import ru.yandex.monitoring.api.v3.project.manager.UpdateProjectRequest;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.gateway.api.v3.utils.ProtoJsonUtils;
import ru.yandex.solomon.util.Proto;

/**
 * @author Alexey Trushkin
 */
public class ProjectManagerClient {

    public static void main(String[] args) throws Exception {
        AuthCallCredentials credentials = AuthCallCredentials.asUser("alextrushkin");
        var channel = localChannel();
        projectApi(credentials, channel);
        accessApi(credentials, channel);
    }

    private static void accessApi(AuthCallCredentials credentials, ManagedChannel channel) {
        var client = AccessServiceGrpc.newBlockingStub(channel).withCallCredentials(credentials);

        System.out.println(ProtoJsonUtils.toJson(AuthorizeRequest.newBuilder()
                .setSubject(Subject.newBuilder()
                        .setOAuthSubject(Subject.OAuthSubject.newBuilder()
                                .setLogin("alextrushkin")
                                .build())
                        .build())
                .setPermission(Permission.ALERT_MANAGEMENT.getSlug())
                .setResourcePath(ResourcePath.newBuilder()
                        .addPath(Resource.newBuilder()
                                .setId("managed-postgresql")
                                .setType("project-manager.service-provider")
                                .build())
                        .addPath(Resource.newBuilder()
                                .setId("managed-postgresql")
                                .setType("project-manager.project")
                                .build())
                        .build())
                .build(), true));
        client.authorize(AuthorizeRequest.newBuilder()
                .setSubject(Subject.newBuilder()
                        .setOAuthSubject(Subject.OAuthSubject.newBuilder()
                                .setLogin("alextrushkin")
                                .build())
                        .build())
                .setPermission(Permission.ALERT_MANAGEMENT.getSlug())
                .setResourcePath(ResourcePath.newBuilder()
                        .addPath(Resource.newBuilder()
                                .setId("managed-postgresql")
                                .setType("project-manager.service-provider")
                                .build())
                        .addPath(Resource.newBuilder()
                                .setId("managed-postgresql")
                                .setType("project-manager.project")
                                .build())
                        .build())
                .build());

        client.authorize(AuthorizeRequest.newBuilder()
                .setSubject(Subject.newBuilder()
                        .setOAuthSubject(Subject.OAuthSubject.newBuilder()
                                .setLogin("alextrushkin")
                                .build())
                        .build())
                .setPermission(Permission.CONFIGS_UPDATE.getSlug())
                .setResourcePath(ResourcePath.newBuilder()
                        .addPath(Resource.newBuilder()
                                .setId("solomo1n")
                                .setType("project-manager.project")
                                .build())
                        .build())
                .build());


        var result = client.availableResources(AvailableResourcesRequest.newBuilder()
                .addRoles(Role.ROLE_PROJECT_ADMIN)
                .setSubject(Subject.newBuilder()
                        .setAnonymousAccount(Subject.AnonymousSubject.newBuilder().build())
                        .build())
                .build());
        System.out.println("AnonymousSubject:\n" + result);
        result = client.availableResources(AvailableResourcesRequest.newBuilder()
                .addRoles(Role.ROLE_PROJECT_ADMIN)
                .setSubject(Subject.newBuilder()
                        .setIamServiceSubject(Subject.IamServiceSubject.newBuilder()
                                .setId("alextrushkin")
                                .build())
                        .build())
                .build());
        System.out.println("IamServiceSubject:\n" + result);
        result = client.availableResources(AvailableResourcesRequest.newBuilder()
                .addRoles(Role.ROLE_PROJECT_ADMIN)
                .setSubject(Subject.newBuilder()
                        .setIamUserSubject(Subject.IamUserSubject.newBuilder()
                                .setId("alextrushkin")
                                .build())
                        .build())
                .build());
        System.out.println("IamUserSubject:\n" + result);
        result = client.availableResources(AvailableResourcesRequest.newBuilder()
                .addRoles(Role.ROLE_PROJECT_ADMIN)
                .setSubject(Subject.newBuilder()
                        .setTvmUserSubject(Subject.TvmUserSubject.newBuilder()
                                .setLogin("alextrushkin")
                                .setUid(123)
                                .build())
                        .build())
                .build());
        System.out.println("TvmUserSubject:\n" + result);
        result = client.availableResources(AvailableResourcesRequest.newBuilder()
                .addRoles(Role.ROLE_PROJECT_ADMIN)
                .setSubject(Subject.newBuilder()
                        .setTvmServiceSubject(Subject.TvmServiceSubject.newBuilder()
                                .setClientId(123)
                                .build())
                        .build())
                .build());
        System.out.println("TvmServiceSubject:\n" + result);
        result = client.availableResources(AvailableResourcesRequest.newBuilder()
                .addRoles(Role.ROLE_PROJECT_ADMIN)
                .setSubject(Subject.newBuilder()
                        .setOAuthSubject(Subject.OAuthSubject.newBuilder()
                                .setLogin("alextrushkin")
                                .setUid(123)
                                .build())
                        .build())
                .build());
        System.out.println("OAuthSubject:\n" + result);
        result = client.availableResources(AvailableResourcesRequest.newBuilder()
                .addRoles(Role.ROLE_PROJECT_ADMIN)
                .setSubject(Subject.newBuilder()
                        .setOpenIdSubject(Subject.OpenIdSubject.newBuilder()
                                .setLogin("alextrushkin")
                                .setAccountId("alextrushkin")
                                .build())
                        .build())
                .build());
        System.out.println("OpenIdSubject:\n" + result);
        result = client.availableResources(AvailableResourcesRequest.newBuilder()
                .addRoles(Role.ROLE_PROJECT_ADMIN)
                .setSubject(Subject.newBuilder()
                        .setSessionIdCookieSubject(Subject.SessionIdCookieSubject.newBuilder()
                                .setLogin("alextrushkin")
                                .build())
                        .build())
                .build());
        System.out.println("SessionIdCookieSubject:\n" + result);
    }

    private static void projectApi(AuthCallCredentials credentials, ManagedChannel channel) {
        var client = ProjectServiceGrpc.newBlockingStub(channel).withCallCredentials(credentials);
        var operation = client.create(CreateProjectRequest.newBuilder()
                .setProjectId(UUID.randomUUID().toString())
                .setName("alextrushkin_tests")
                .setDescription(UUID.randomUUID().toString())
                .setAbcService("solomon")
                .putAllLabels(Map.of(UUID.randomUUID().toString(), UUID.randomUUID().toString()))
                .build());
        if (operation.hasError()) {
            throw new IllegalArgumentException(operation.getError().getMessage());
        }
        var createdProject = Proto.unpack(operation.getResponse(), Project.class);
        operation = client.update(UpdateProjectRequest.newBuilder()
                .setProjectId(createdProject.getProjectId())
                .setName(createdProject.getName() + "new")
                .setDescription(createdProject.getDescription() + "new")
                .setAbcService(createdProject.getAbcService())
                .putAllLabels(createdProject.getLabelsMap())
                .putLabels("1", "2")
                .setEtag(createdProject.getEtag())
                .build());

        if (operation.hasError()) {
            throw new IllegalArgumentException(operation.getError().getMessage());
        }
        createdProject = Proto.unpack(operation.getResponse(), Project.class);

        var project = client.get(GetProjectRequest.newBuilder()
                .setProjectId(createdProject.getProjectId())
                .build());
        System.out.println(project);

        var operations = client.listOperations(ListProjectOperationsRequest.newBuilder()

                .build());
        System.out.println("operations: " + operations);

        operation = client.delete(DeleteProjectRequest.newBuilder()
                .setProjectId(project.getProjectId())
                .build());

        if (operation.hasError()) {
            throw new IllegalArgumentException(operation.getError().getMessage());
        }
        System.out.println("deleteOperation: " + operation);

        var list = client.list(ListProjectsRequest.newBuilder()
                .setFilterByAbcService("solomon")
                .addFilterByRole(Role.ROLE_PROJECT_ADMIN)
                .build());
        System.out.println(list);

        list = client.list(ListProjectsRequest.newBuilder()
                .setPageSize(2)
                .build());
        System.out.println(list);

        list = client.list(ListProjectsRequest.newBuilder()
                .setPageSize(2)
                .setPageToken(list.getNextPageToken())
                .build());
        System.out.println(list);
    }

    private static ManagedChannel localChannel() {
        return NettyChannelBuilder.forTarget("localhost:5780")
                .negotiationType(NegotiationType.PLAINTEXT)
                .build();
    }


    private static ManagedChannel remoteChannel(String address) throws SSLException {
        if (address.isBlank()) {
            address = "pm-pre.yandex-team.ru:443";
        }
        var sslContext = GrpcSslContexts.forClient().trustManager(new File("/Users/alextrushkin/arcadia/certs/cacert.pem")).build();
        return NettyChannelBuilder.forTarget(address)
                .negotiationType(NegotiationType.TLS)
                .sslContext(sslContext)
                .build();
    }
}
