package ru.yandex.solomon.experiments.alextrushkin;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.net.ssl.SSLException;

import io.grpc.ManagedChannel;
import io.grpc.netty.GrpcSslContexts;
import io.grpc.netty.NegotiationType;
import io.grpc.netty.NettyChannelBuilder;
import yandex.cloud.priv.team.integration.v1.AbcServiceGrpc;
import yandex.cloud.priv.team.integration.v1.PTIAS;

import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;

/**
 * @author Alexey Trushkin
 */
public class ResolveCloudToAbc implements AutoCloseable {

    private final String ytTable;
    private final String ytAddress;
    private static String TOKEN = System.getenv("IAM_TOKEN");

    public ResolveCloudToAbc(String ytTable, String yt) {
        this.ytTable = ytTable;
        this.ytAddress = yt;
    }

    private void exportYtTable() throws SSLException {
        Yt yt = YtUtils.http(ytAddress);
        YPath cloudsTable = YPath.simple("//home/cloud-dwh/data/prod_internal/ods/mdb/clouds");
        Map<Integer, Cloud> cloudMap = new HashMap<>();
        yt.tables().read(
                cloudsTable.withColumns("cloud_ext_id", "cloud_id"),
                YTableEntryTypes.YSON,
                (entry) -> {
                    var cloud_ext_id = entry.getString("cloud_ext_id");
                    var cloud_id = entry.getInt("cloud_id");
                    cloudMap.put(cloud_id, new Cloud(cloud_id, cloud_ext_id));
                }
        );

        var channel = remoteChannel("");
        List<YTreeMapNode> rows = new ArrayList<>();
        for (var cloud : cloudMap.values()) {
            var abc = resolveAbc(channel, cloud);
            if (abc == null) {
                continue;
            }
            rows.add(createEntry(cloud, abc));
        }

        YPath table = YPath.simple(ytTable);
        yt.tables().write(table, YTableEntryTypes.YSON, rows);
        System.out.println("Exported " + rows.size());
    }

    private YTreeMapNode createEntry(Cloud cloud, PTIAS.ResolveResponse abc) {
        return YTree.mapBuilder()
                .key("cloud_ext_id").value(cloud.cloudExtId)
                .key("cloud_id").value(cloud.cloudId)
                .key("abc_slug").value(abc.getAbcSlug())
                .key("abc_id").value(abc.getAbcId())
                .buildMap();
    }

    private PTIAS.ResolveResponse resolveAbc(ManagedChannel channel, Cloud cloud) {
        var s = cacheAbc.get(cloud.cloudId);
        if (s != null) {
            return null;
        }
        AuthCallCredentials credentials = AuthCallCredentials.iam(TOKEN);
        var client = AbcServiceGrpc.newBlockingStub(channel).withCallCredentials(credentials);
        try {
            final PTIAS.ResolveResponse resolve = client.resolve(PTIAS.ResolveRequest.newBuilder()
                    .setCloudId(cloud.cloudExtId())
                    .build());
            cacheAbc.put(cloud.cloudId, resolve);
            return resolve;
        } catch (Exception e) {
            System.out.println(e.getMessage());
            return null;
        }
    }

    private Map<Integer, PTIAS.ResolveResponse> cacheAbc = new HashMap<>();

    public static void main(String... args) {
        var table = "//home/solomon/service_provider_alerts/service_provider_exports/cloud_to_abc";
        var yt = "hahn.yt.yandex.net";
        System.out.println("Start resolve clouds " + yt + table);
        try (var cli = new ResolveCloudToAbc(table, yt)) {
            cli.exportYtTable();
        } catch (Throwable t) {
            t.printStackTrace();
            System.exit(1);
        }
        System.out.println("Finished");
        System.exit(0);
    }

    @Override
    public void close() {

    }

    private static record Cloud(int cloudId, String cloudExtId) {

    }

    private static ManagedChannel remoteChannel(String address) throws SSLException {
        if (address.isBlank()) {
            address = "ti.cloud.yandex-team.ru:443";
        }
        var sslContext = GrpcSslContexts.forClient().trustManager(new File("/Users/alextrushkin/arcadia/certs/cacert.pem")).build();
        return NettyChannelBuilder.forTarget(address)
                .negotiationType(NegotiationType.TLS)
                .sslContext(sslContext)
                .build();
    }
}
