package ru.yandex.solomon.experiments.benchmark;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.model.point.AggrPointData;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.stockpile.api.EDecimPolicy;
import ru.yandex.stockpile.memState.MetricIdAndData;
import ru.yandex.stockpile.server.shard.MergeKind;
import ru.yandex.stockpile.server.shard.MergeProcessMetrics;
import ru.yandex.stockpile.server.shard.MergeTask;
import ru.yandex.stockpile.server.shard.MergeTaskResult;

/*

Benchmark                    (archiveCount)    (kind)  (pointIntoArchive)  Mode  Cnt   Score   Error  Units
MergeTaskBenchmark.mergeNew               1  ETERNITY                 500  avgt   10   0.039 ± 0.001  ms/op
MergeTaskBenchmark.mergeNew               1  ETERNITY                5000  avgt   10   0.371 ± 0.018  ms/op
MergeTaskBenchmark.mergeNew              10  ETERNITY                 500  avgt   10   1.440 ± 0.011  ms/op
MergeTaskBenchmark.mergeNew              10  ETERNITY                5000  avgt   10  14.944 ± 0.714  ms/op
MergeTaskBenchmark.mergeOld               1  ETERNITY                 500  avgt   10   0.053 ± 0.001  ms/op
MergeTaskBenchmark.mergeOld               1  ETERNITY                5000  avgt   10   0.520 ± 0.030  ms/op
MergeTaskBenchmark.mergeOld              10  ETERNITY                 500  avgt   10   6.667 ± 0.184  ms/op
MergeTaskBenchmark.mergeOld              10  ETERNITY                5000  avgt   10  74.024 ± 2.997  ms/op

 */

/**
 * @author Vladimir Gordiychuk
 */
@Fork(value = 1)
@Measurement(iterations = 10, time = 1, timeUnit = TimeUnit.SECONDS)
@Warmup(iterations = 5, time = 1, timeUnit = TimeUnit.SECONDS)
@State(Scope.Thread)
@Threads(1) //current test not support concurrent execution
@BenchmarkMode(Mode.AverageTime)
@OutputTimeUnit(TimeUnit.MILLISECONDS)
public class MergeTaskBenchmark {
    @Param({"1", "10"})
    private int archiveCount;

    @Param({"500", "5000"})
    private int pointIntoArchive;

    @Param({"DAILY", "ETERNITY"})
    private MergeKind kind;

    private MergeTask mergeTaskNew;

    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
                .include(MergeTaskBenchmark.class.getName())
                .detectJvmArgs()
                .jvmArgs("-Xmx3g", "-Xms3g")
//                .addProfiler(FlightRecorderProfiler.class)
                .build();

        new Runner(opt).run();
    }

    @Setup
    public void setUp() {
        int mask = StockpileColumn.TS.mask() | StockpileColumn.VALUE.mask();
        AggrGraphDataArrayList source = new AggrGraphDataArrayList(mask, pointIntoArchive);
        long now = System.currentTimeMillis();

        long stepMillis = TimeUnit.SECONDS.toMillis(15L);
        AggrPointData point = new AggrPointData();

        List<MetricIdAndData> mergeItems = new ArrayList<>(archiveCount);
        for (int indexArchive = 0; indexArchive < archiveCount; indexArchive++) {
            for (int indexPoint = 0; indexPoint < pointIntoArchive; indexPoint++) {
                now += stepMillis;
                point.tsMillis = now;
                point.valueNum = indexPoint;
                source.addRecordData(mask, point);
            }

            MetricArchiveMutable archive = MetricArchiveMutable.of(source);
            archive.setDecimPolicyId((short) EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber());
            mergeItems.add(new MetricIdAndData(1, archive.getLastTsMillis(), 0, archive.toImmutableNoCopy()));
        }

        MergeProcessMetrics.MergeTaskMetrics taskMetrics = new MergeProcessMetrics.MergeTaskMetrics(kind.name());

        now += TimeUnit.DAYS.toMillis(10);

        boolean allowDecim = kind == MergeKind.ETERNITY;
        mergeTaskNew = new MergeTask(1, mergeItems, now, 0, allowDecim, taskMetrics);
    }

    @Benchmark
    public MergeTaskResult mergeNew() {
        return mergeTaskNew.run();
    }
}
