package ru.yandex.solomon.experiments.benchmark;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.io.Resources;
import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Level;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.solomon.gateway.api.internal.dto.StockpileDataResultDto;
import ru.yandex.solomon.model.type.Histogram;
import ru.yandex.solomon.model.type.ugram.Ugram;

/**
 * @author Vladimir Gordiychuk
 */
@Fork(value = 1)
@Measurement(iterations = 10, time = 2, timeUnit = TimeUnit.SECONDS)
@Warmup(iterations = 5, time = 2, timeUnit = TimeUnit.SECONDS)
@State(Scope.Thread)
@Threads(1) //current test not support concurrent execution
@BenchmarkMode(Mode.AverageTime)
@OutputTimeUnit(TimeUnit.MILLISECONDS)
public class UgramMergeJmh {
    @Param({
        "cacher-reader.stockpile_sensors_per_request_hgram.json",
        "rtc_unistat-get-generic-handler-time_hgram.json",
        "rtc_unistat-getting-pinger-status-update_hgram.json",
        "stream_nginxtskv-get-hit-request-time_hgram.json",
        "unistat-kz-abs_dhhh.json",
    })
    public String dataset;
    private List<Histogram> snapshots;
    private Ugram ugram;

    /*

Benchmark                                                            (dataset)  Mode  Cnt  Score   Error  Units
UgramMergeJmh.mergeOld  cacher-reader.stockpile_sensors_per_request_hgram.json  avgt   10  0.048 ± 0.005  ms/op
UgramMergeJmh.mergeOld         rtc_unistat-get-generic-handler-time_hgram.json  avgt   10  0.226 ± 0.019  ms/op
UgramMergeJmh.mergeOld     rtc_unistat-getting-pinger-status-update_hgram.json  avgt   10  8.833 ± 0.497  ms/op
UgramMergeJmh.mergeOld        stream_nginxtskv-get-hit-request-time_hgram.json  avgt   10  0.071 ± 0.005  ms/op
UgramMergeJmh.mergeOld                                unistat-kz-abs_dhhh.json  avgt   10  0.077 ± 0.006  ms/op

     */

    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
            .include(UgramMergeJmh.class.getName())
            .detectJvmArgs()
            .jvmArgs("-Xmx1g", "-Xms1g")
//            .addProfiler(GCProfiler.class)
//            .addProfiler(FlightRecorderProfiler.class)
            .build();

        new Runner(opt).run();
    }

    @Setup
    public void setUp() throws IOException {
        var dataSetJson = readResource(this.dataset);

        ObjectMapper mapper = new ObjectMapper();
        var r = mapper.readValue(dataSetJson, StockpileDataResultDto.class);
        snapshots = r.points.stream()
            .map(point -> point.histogram)
            .map(hist -> Histogram.newInstance().copyFrom(hist.bounds, hist.buckets))
            .collect(Collectors.toList());
        ugram = Ugram.create();
    }

    @Setup(Level.Invocation)
    public void beforeTest() {
        ugram.reset();
    }

    private String readResource(String name) {
        try {
            return Resources.toString(UgramMergeJmh.class.getResource("data/ugram/" + name), StandardCharsets.UTF_8);
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @Benchmark
    public Object mergeOld() {
        for (var snapshot : snapshots) {
            ugram.merge(snapshot);
        }
        return ugram;
    }

//    @Benchmark
//    public Object mergeNew() {
//        var ugram = Ugram.create();
//        for (var snapshot : snapshots) {
//            ugram.mergeNew(snapshot);
//        }
//        return ugram;
//    }
}
