package ru.yandex.solomon.experiments.benchmark;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.io.Resources;
import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Level;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.solomon.gateway.api.internal.dto.StockpileDataResultDto;
import ru.yandex.solomon.model.type.Histogram;
import ru.yandex.solomon.model.type.MutableHistogram;
import ru.yandex.solomon.model.type.ugram.Ugram;

/**
 * @author Vladimir Gordiychuk
 */
@Fork(value = 1)
@Measurement(iterations = 10, time = 2, timeUnit = TimeUnit.SECONDS)
@Warmup(iterations = 5, time = 2, timeUnit = TimeUnit.SECONDS)
@State(Scope.Thread)
@Threads(1) //current test not support concurrent execution
@BenchmarkMode(Mode.AverageTime)
@OutputTimeUnit(TimeUnit.MILLISECONDS)
public class UgramVsHistCompare {
    @Param({
        "cacher-reader.stockpile_sensors_per_request_hgram.json",
        "rtc_unistat-get-generic-handler-time_hgram.json",
        "rtc_unistat-getting-pinger-status-update_hgram.json",
        "stream_nginxtskv-get-hit-request-time_hgram.json",
        "unistat-kz-abs_dhhh.json",
    })
    public String dataset;
    private List<Histogram> snapshots;
    private Ugram ugram;
    private MutableHistogram hist;

    /*

Benchmark                                                                   (dataset)  Mode  Cnt  Score   Error  Units
UgramVsHistCompare.histMerge   cacher-reader.stockpile_sensors_per_request_hgram.json  avgt   10  0.021 ± 0.002  ms/op
UgramVsHistCompare.histMerge          rtc_unistat-get-generic-handler-time_hgram.json  avgt   10  0.257 ± 0.016  ms/op
UgramVsHistCompare.histMerge      rtc_unistat-getting-pinger-status-update_hgram.json  avgt   10  5.248 ± 1.347  ms/op
UgramVsHistCompare.histMerge         stream_nginxtskv-get-hit-request-time_hgram.json  avgt   10  0.038 ± 0.004  ms/op
UgramVsHistCompare.histMerge                                 unistat-kz-abs_dhhh.json  avgt   10  0.034 ± 0.003  ms/op
UgramVsHistCompare.ugramMerge  cacher-reader.stockpile_sensors_per_request_hgram.json  avgt   10  0.047 ± 0.009  ms/op
UgramVsHistCompare.ugramMerge         rtc_unistat-get-generic-handler-time_hgram.json  avgt   10  0.250 ± 0.015  ms/op
UgramVsHistCompare.ugramMerge     rtc_unistat-getting-pinger-status-update_hgram.json  avgt   10  8.633 ± 0.700  ms/op
UgramVsHistCompare.ugramMerge        stream_nginxtskv-get-hit-request-time_hgram.json  avgt   10  0.079 ± 0.028  ms/op
UgramVsHistCompare.ugramMerge                                unistat-kz-abs_dhhh.json  avgt   10  0.078 ± 0.010  ms/op

     */

    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
            .include(UgramVsHistCompare.class.getName())
            .detectJvmArgs()
            .jvmArgs("-Xmx1g", "-Xms1g")
//            .addProfiler(GCProfiler.class)
//            .addProfiler(FlightRecorderProfiler.class)
            .build();

        new Runner(opt).run();
    }

    @Setup
    public void setUp() throws IOException {
        var dataSetJson = readResource(this.dataset);

        ObjectMapper mapper = new ObjectMapper();
        var r = mapper.readValue(dataSetJson, StockpileDataResultDto.class);
        snapshots = r.points.stream()
            .map(point -> point.histogram)
            .map(hist -> Histogram.newInstance().copyFrom(hist.bounds, hist.buckets))
            .collect(Collectors.toList());
        ugram = Ugram.create();
        hist = new MutableHistogram();
    }

    @Setup(Level.Invocation)
    public void beforeTest() {
        ugram.reset();
        hist.reset();
    }

    private String readResource(String name) {
        try {
            return Resources.toString(UgramMergeJmh.class.getResource("data/ugram/" + name), StandardCharsets.UTF_8);
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @Benchmark
    public Object ugramMerge() {
        for (var snapshot : snapshots) {
            ugram.merge(snapshot);
        }
        return ugram.snapshot(null);
    }

    @Benchmark
    public Object histMerge() {
        for (var snapshot : snapshots) {
            hist.addHistogram(snapshot);
        }
        return hist.snapshot();
    }
}
