package ru.yandex.solomon.experiments.egorlitvinenk;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.locks.LockSupport;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Level;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.misc.concurrent.CompletableFutures;

/**
 *
 * @author Egor Litvinenko
 */
@Fork(value = 2)
@Measurement(iterations = 5, time = 5, timeUnit = TimeUnit.SECONDS)
@Warmup(iterations = 3, time = 2, timeUnit = TimeUnit.SECONDS)
@State(Scope.Thread)
@Threads(4)
@BenchmarkMode({ Mode.SampleTime })
@OutputTimeUnit(TimeUnit.MILLISECONDS)
public class FuturesAccumulateBenchmark {

    @Param({
//            "1",
//            "4",
//            "10",
            "100",
//            "1000"
    })
    public int taskCount;

    public List<CompletableFuture<Integer>> futureList;
    private final Function<Throwable, Integer> onException = ex -> null;
    private final BinaryOperator<Integer> reducer = Integer::sum;
    private final Predicate<Integer> testValue = i -> Integer.MAX_VALUE != i;
    private final ForkJoinPool pool = new ForkJoinPool();


    @Setup(Level.Invocation)
    public void setUp() throws IOException {
        int max = taskCount;
        futureList = IntStream.range(0, max).mapToObj(
                i -> CompletableFuture.supplyAsync(() -> {
                    var pause = TimeUnit.MILLISECONDS.toNanos(ThreadLocalRandom.current().nextInt(10));
                    try {
                        LockSupport.parkNanos(pause);
                    } catch (Exception e) {}
                    return ThreadLocalRandom.current().nextInt(max);
                })
        ).collect(Collectors.toList());
    }

    @Benchmark
    public Integer accumulate(FuturesAccumulateBenchmark benchmark) throws Exception {
        try {
            return CompletableFutures.accumulateWithCancellation(
                    benchmark.futureList,
                    benchmark.testValue,
                    benchmark.reducer,
                    benchmark.onException,
                    benchmark.pool
            ).get(1_000, TimeUnit.MILLISECONDS);
        } catch (TimeoutException e) {
            return Integer.MIN_VALUE;
        }
    }

    @Benchmark
    public Integer allOf(FuturesAccumulateBenchmark benchmark) throws Exception {
        try {
            return CompletableFutures.allOf(benchmark.futureList)
                    .thenApplyAsync(
                            list -> list.stream().reduce(benchmark.reducer).orElse(Integer.MAX_VALUE),
                            benchmark.pool
                    ).get(1_000, TimeUnit.MILLISECONDS);
        } catch (TimeoutException e) {
            return Integer.MIN_VALUE;
        }
    }

    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
                .include(FuturesAccumulateBenchmark.class.getName())
                .detectJvmArgs()
                .jvmArgs("-Xmx1g", "-Xms1g", "--enable-preview")
//            .addProfiler(GCProfiler.class)
//            .addProfiler(FlightRecorderProfiler.class)
                .build();

        new Runner(opt).run();
    }


//    It is not expected to have big difference

//    Without pause
//    Benchmark                                                 (taskCount)    Mode      Cnt   Score    Error  Units
//    FuturesAccumulateBenchmark.accumulate                               4  sample  4817000   0.013 ±  0.001  ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.00              4  sample            0.001           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.50              4  sample            0.009           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.90              4  sample            0.018           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.95              4  sample            0.030           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.99              4  sample            0.059           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.999             4  sample            0.161           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.9999            4  sample            7.211           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p1.00              4  sample           27.853           ms/op
//    FuturesAccumulateBenchmark.accumulate                              10  sample  3274237   0.020 ±  0.001  ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.00             10  sample            0.001           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.50             10  sample            0.012           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.90             10  sample            0.031           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.95             10  sample            0.048           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.99             10  sample            0.099           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.999            10  sample            0.428           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.9999           10  sample           10.011           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p1.00             10  sample           37.356           ms/op
//    FuturesAccumulateBenchmark.accumulate                             100  sample  2959030   0.047 ±  0.001  ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.00            100  sample            0.010           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.50            100  sample            0.043           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.90            100  sample            0.070           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.95            100  sample            0.080           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.99            100  sample            0.104           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.999           100  sample            0.239           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.9999          100  sample            3.916           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p1.00            100  sample           21.299           ms/op
//    FuturesAccumulateBenchmark.allOf                                    4  sample  4149752   0.015 ±  0.001  ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.00                        4  sample           ≈ 10⁻³           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.50                        4  sample            0.010           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.90                        4  sample            0.021           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.95                        4  sample            0.033           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.99                        4  sample            0.062           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.999                       4  sample            0.217           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.9999                      4  sample            8.618           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p1.00                        4  sample           20.218           ms/op
//    FuturesAccumulateBenchmark.allOf                                   10  sample  2677262   0.018 ±  0.001  ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.00                       10  sample            0.001           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.50                       10  sample            0.012           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.90                       10  sample            0.028           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.95                       10  sample            0.040           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.99                       10  sample            0.075           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.999                      10  sample            0.287           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.9999                     10  sample            9.994           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p1.00                       10  sample           14.909           ms/op
//    FuturesAccumulateBenchmark.allOf                                  100  sample  3065928   0.037 ±  0.001  ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.00                      100  sample            0.003           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.50                      100  sample            0.033           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.90                      100  sample            0.058           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.95                      100  sample            0.067           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.99                      100  sample            0.088           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.999                     100  sample            0.143           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.9999                    100  sample            2.646           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p1.00                      100  sample           10.060           ms/op
//    FuturesAccumulateBenchmark.accumulate                            1000  sample  282826    0.441 ± 0.007  ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.00           1000  sample            0.074          ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.50           1000  sample            0.296          ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.90           1000  sample            0.569          ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.95           1000  sample            0.880          ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.99           1000  sample            3.789          ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.999          1000  sample           13.929          ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.9999         1000  sample           40.661          ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p1.00           1000  sample          114.033          ms/op
//    FuturesAccumulateBenchmark.allOf                                 1000  sample  554818    0.173 ± 0.001  ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.00                     1000  sample            0.023          ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.50                     1000  sample            0.141          ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.90                     1000  sample            0.316          ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.95                     1000  sample            0.376          ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.99                     1000  sample            0.505          ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.999                    1000  sample            2.295          ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.9999                   1000  sample            6.320          ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p1.00                     1000  sample           13.173          ms/op
//    Process finished with exit code 0

//    With pause and 1sec timeout
//    Benchmark                                                 (taskCount)    Mode  Cnt     Score    Error  Units
//    FuturesAccumulateBenchmark.accumulate                             100  sample  320   616.568 ± 30.169  ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.00            100  sample        212.861           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.50            100  sample        595.067           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.90            100  sample        853.226           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.95            100  sample        994.732           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.99            100  sample       1004.536           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.999           100  sample       1004.536           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p0.9999          100  sample       1004.536           ms/op
//    FuturesAccumulateBenchmark.accumulate:accumulate·p1.00            100  sample       1004.536           ms/op
//    FuturesAccumulateBenchmark.allOf                                  100  sample  322   613.241 ± 31.108  ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.00                      100  sample        187.171           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.50                      100  sample        593.494           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.90                      100  sample        834.666           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.95                      100  sample        998.087           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.99                      100  sample       1004.536           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.999                     100  sample       1004.536           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p0.9999                    100  sample       1004.536           ms/op
//    FuturesAccumulateBenchmark.allOf:allOf·p1.00                      100  sample       1004.536           ms/op


}
