package ru.yandex.solomon.experiments.gordiychuk;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.atomic.AtomicInteger;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.HashBasedTable;
import com.google.common.collect.Table;
import com.google.common.net.HostAndPort;
import com.yandex.ydb.table.query.Params;
import org.apache.commons.lang3.math.NumberUtils;

import ru.yandex.solomon.core.db.dao.ydb.YdbClustersDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbServicesDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbShardsDao;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;

import static com.yandex.ydb.table.values.PrimitiveValue.utf8;

/**
 * @author Vladimir Gordiychuk
 */
public class ActualizePushShardIds {
    private static final String YDB_HOST = "127.0.0.55";
    private static final String DB_PREFIX = "/pre-prod_global/solomon";

    private static YdbClient ydb;
    private static YdbShardsDao shardsDao;
    private static YdbClustersDao clustersDao;
    private static YdbServicesDao servicesDao;

    public static void main(String[] args) {
        try (var client = YdbHelper.createYdbClient(HostAndPort.fromParts(YDB_HOST, 2135))) {
            ydb = client;
            shardsDao = new YdbShardsDao(client.table, DB_PREFIX + "/Config/V2/Shard", new ObjectMapper(), ForkJoinPool.commonPool());
            clustersDao = new YdbClustersDao(client.table, DB_PREFIX + "/Config/V2/Cluster", new ObjectMapper(), ForkJoinPool.commonPool());
            servicesDao = new YdbServicesDao(client.table, DB_PREFIX + "/Config/V2/Service", new ObjectMapper(), ForkJoinPool.commonPool());

            updateShardsClusters();
            updateShardServices();
            rmEmptyClusters();
            rmEmptyServices();
            System.exit(0);
        }
    }

    public static void updateShardsClusters() {
        System.out.println("Update Shard Clusters");
        var clusters = readClusters().join();
        var shards = readShards().join();
        List<Shard> updateShards = new ArrayList<>();
        var it = shards.rowKeySet().iterator();
        while (it.hasNext()) {
            String projectId = it.next();
            var projectClusters = clusters.row(projectId);
            var projectShards = shards.row(projectId);
            for (var shard : projectShards.values()) {
                var cluster = projectClusters.get(shard.getClusterId());
                if (cluster == null) {
                    continue;
                }

                if (!isBadId(cluster.getProjectId(), cluster.getName(), cluster.getId())) {
                    continue;
                }

                var goodCluster = projectClusters.get(goodId(projectId, cluster.getName()));
                if (goodCluster == null) {
                    continue;
                }

                var copy = goodCluster.toBuilder()
                        .setId(cluster.getId())
                        .setVersion(cluster.getVersion())
                        .setCreatedAt(cluster.getCreatedAt())
                        .setCreatedBy(cluster.getCreatedBy())
                        .setUpdatedAt(cluster.getUpdatedAt())
                        .setUpdatedBy(cluster.getUpdatedBy())
                        .build();

                if (!cluster.equals(copy)) {
                    continue;
                }

                System.out.println("update shardId " + shard.getId());
                updateShards.add(shard);
            }
        }
        System.out.println("Shards to update: " + updateShards.size());
        var shardsIt = updateShards.iterator();
        AtomicInteger counter = new AtomicInteger();
        AsyncActorBody body = () -> {
            if (!shardsIt.hasNext()) {
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }

            var shard = shardsIt.next();
            System.out.println(counter.incrementAndGet() + " " + shard.getId());
            String query = "--!syntax_v1\n" +
                    "DECLARE $projectId AS Utf8;\n" +
                    "DECLARE $shardId as Utf8;\n" +
                    "DECLARE $clusterId as Utf8;\n" +
                    "UPDATE `" + DB_PREFIX + "/Config/V2/Shard` " +
                    "SET clusterId = $clusterId " +
                    "WHERE projectId = $projectId " +
                    "AND id = $shardId;";
            var params = Params.of(
                    "$projectId", utf8(shard.getProjectId()),
                    "$shardId", utf8(shard.getId()),
                    "$clusterId", utf8(goodId(shard.getProjectId(), shard.getClusterName()))
            );
            return ydb.fluent().execute(query, params)
                    .thenAccept(r -> r.expect("success"));
        };
        var runner = new AsyncActorRunner(body, ForkJoinPool.commonPool(), 50);
        runner.start().join();
    }

    private static void updateShardServices() {
        System.out.println("Update Shard Services");
        var services = readServices().join();
        var shards = readShards().join();
        List<Shard> updateShards = new ArrayList<>();
        var it = shards.rowKeySet().iterator();
        while (it.hasNext()) {
            String projectId = it.next();
            var projectServices = services.row(projectId);
            var projectShards = shards.row(projectId);
            for (var shard : projectShards.values()) {
                var service = projectServices.get(shard.getServiceId());
                if (service == null) {
                    continue;
                }

                if (!isBadId(service.getProjectId(), service.getName(), service.getId())) {
                    continue;
                }

                var goodService = projectServices.get(goodId(projectId, service.getName()));
                if (goodService == null) {
                    continue;
                }

                var copy = goodService.toBuilder()
                        .setId(service.getId())
                        .setVersion(service.getVersion())
                        .setCreatedAt(service.getCreatedAt())
                        .setCreatedBy(service.getCreatedBy())
                        .setUpdatedAt(service.getUpdatedAt())
                        .setUpdatedBy(service.getUpdatedBy())
                        .build();

                if (!service.equals(copy)) {
                    continue;
                }

                System.out.println("update shardId " + shard.getId());
                updateShards.add(shard);
            }
        }
        System.out.println("Shards to update: " + updateShards.size());
        var shardsIt = updateShards.iterator();
        AtomicInteger counter = new AtomicInteger();
        AsyncActorBody body = () -> {
            if (!shardsIt.hasNext()) {
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }

            var shard = shardsIt.next();
            System.out.println(counter.incrementAndGet() + " " + shard.getId());
            String query = "--!syntax_v1\n" +
                    "DECLARE $projectId AS Utf8;\n" +
                    "DECLARE $shardId as Utf8;\n" +
                    "DECLARE $serviceId as Utf8;\n" +
                    "UPDATE `" + DB_PREFIX + "/Config/V2/Shard` " +
                    "SET serviceId = $serviceId " +
                    "WHERE projectId = $projectId " +
                    "AND id = $shardId;";
            var params = Params.of(
                    "$projectId", utf8(shard.getProjectId()),
                    "$shardId", utf8(shard.getId()),
                    "$serviceId", utf8(goodId(shard.getProjectId(), shard.getServiceName()))
            );
            return ydb.fluent().execute(query, params)
                    .thenAccept(r -> r.expect("success"));
        };
        var runner = new AsyncActorRunner(body, ForkJoinPool.commonPool(), 50);
        runner.start().join();
    }

    private static void rmEmptyClusters() {
        System.out.println("Rm empty Clusters");
        var clusters = readClusters().join();
        var shards = readShards().join();

        List<Cluster> emptyClusters = new ArrayList<>();
        var it = shards.rowKeySet().iterator();
        while (it.hasNext()) {
            String projectId = it.next();
            var projectClusters = clusters.row(projectId);
            var projectShards = shards.row(projectId);
            for (var shard : projectShards.values()) {
                projectClusters.remove(shard.getClusterId());
            }
            if (projectClusters.isEmpty()) {
                continue;
            }
            System.out.println("Empty clusters at " + projectId + ": " + projectClusters.keySet());
            projectClusters.values()
                    .stream()
                    .filter(cluster -> isBadId(cluster.getProjectId(), cluster.getName(), cluster.getId()))
                    .forEach(emptyClusters::add);
        }
        System.out.println("Empty clusters: " + emptyClusters.size());
        var clustersIt = emptyClusters.iterator();
        AtomicInteger counter = new AtomicInteger();
        var threshold = Instant.now().minus(4, ChronoUnit.HOURS);
        AsyncActorBody body = () -> {
            while (true) {
                if (!clustersIt.hasNext()) {
                    return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
                }

                var cluster = clustersIt.next();
                System.out.println(counter.incrementAndGet() + " " + cluster.getProjectId() + " " + cluster.getId());
                if (cluster.getCreatedAt().isAfter(threshold)) {
                    continue;
                }

                return clustersDao.deleteOne(cluster.getProjectId(), "", cluster.getId());
            }
        };
        var runner = new AsyncActorRunner(body, ForkJoinPool.commonPool(), 50);
        runner.start().join();
    }

    private static void rmEmptyServices() {
        System.out.println("Rm empty Services");
        List<Service> empty = new ArrayList<>();
        {
            var shards = readShards().join();
            var services = readServices().join();
            var it = shards.rowKeySet().iterator();
            while (it.hasNext()) {
                String projectId = it.next();
                var projectServices = services.row(projectId);
                var projectShards = shards.row(projectId);
                for (var shard : projectShards.values()) {
                    projectServices.remove(shard.getServiceId());
                }
                projectServices.values().removeIf(service -> !isBadId(service.getProjectId(), service.getName(), service.getId()));
                if (projectServices.isEmpty()) {
                    continue;
                }
                System.out.println("Empty services at " + projectId + ": " + projectServices.keySet());
                empty.addAll(projectServices.values());
            }
        }
        System.out.println("Empty services: " + empty.size());
        var it = empty.iterator();
        AtomicInteger counter = new AtomicInteger();
        var threshold = Instant.now().minus(4, ChronoUnit.HOURS);
        AsyncActorBody body = () -> {
            while (true) {
                if (!it.hasNext()) {
                    return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
                }

                var service = it.next();
                System.out.println(counter.incrementAndGet() + " " + service.getProjectId() + " " + service.getId());
                if (service.getCreatedAt().isAfter(threshold)) {
                    continue;
                }

                return servicesDao.deleteOne(service.getProjectId(), "", service.getId());
            }
        };
        var runner = new AsyncActorRunner(body, ForkJoinPool.commonPool(), 50);
        runner.start().join();
    }

    private static boolean isBadId(String projectId, String name, String id) {
        String good = goodId(projectId, name);
        if (!id.startsWith(good + "_")) {
            return false;
        }
        String suffix = id.substring(good.length() + 1);
        return NumberUtils.isDigits(suffix);
    }

    private static String goodId(String projectId, String name) {
        return projectId + "_" + name;
    }

    private static CompletableFuture<Table<String, String, Shard>> readShards() {
        return shardsDao.findAll()
                .thenApply(shards -> {
                    var table = HashBasedTable.<String, String, Shard>create();
                    for (var shard : shards) {
                        table.put(shard.getProjectId(), shard.getId(), shard);
                    }
                    System.out.println("Read shards: " + table.size());
                    return table;
                });
    }

    private static CompletableFuture<Table<String, String, Cluster>> readClusters() {
        return clustersDao.findAll()
                .thenApply(clusters -> {
                    var table = HashBasedTable.<String, String, Cluster>create();
                    for (var cluster : clusters) {
                        table.put(cluster.getProjectId(), cluster.getId(), cluster);
                    }
                    System.out.println("Read clusters: " + table.size());
                    return table;
                });
    }

    private static CompletableFuture<Table<String, String, Service>> readServices() {
        return servicesDao.findAll()
                .thenApply(services -> {
                    var table = HashBasedTable.<String, String, Service>create();
                    for (var service : services) {
                        table.put(service.getProjectId(), service.getId(), service);
                    }
                    System.out.println("Read services: " + table.size());
                    return table;
                });
    }
}
