package ru.yandex.solomon.experiments.gordiychuk;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;

import com.google.common.collect.Iterables;
import com.google.common.net.HostAndPort;

import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.metabase.client.MetabaseClient;
import ru.yandex.metabase.client.MetabaseClientOptions;
import ru.yandex.metabase.client.MetabaseClients;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.metabase.api.protobuf.CreateOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.ResolveOneRequest;
import ru.yandex.solomon.metrics.client.CrossDcMetricsClient;
import ru.yandex.solomon.metrics.client.DcMetricsClient;
import ru.yandex.solomon.metrics.client.FindRequest;
import ru.yandex.solomon.metrics.client.MetabaseClientException;
import ru.yandex.solomon.metrics.client.MetabaseStatus;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.metrics.client.ReadManyRequest;
import ru.yandex.solomon.metrics.client.StockpileClientException;
import ru.yandex.solomon.metrics.client.UniqueLabelsRequest;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.model.StockpileKey;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.protobuf.MetricTypeConverter;
import ru.yandex.solomon.model.type.SummaryDouble;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.tool.cfg.SolomonPorts;
import ru.yandex.solomon.tool.stockpile.StockpileShardWriters;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.StockpileClientOptions;
import ru.yandex.stockpile.client.StockpileClients;
import ru.yandex.stockpile.client.StopStrategies;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
public class AggregationAsSummaryMigration implements AutoCloseable {
    private static final int CHUNK_BYTES_SIZE = 5 << 20; // 5 MiB
    private static final int TARGET_NUM_ID = 991207019;

    private final Source source;
    private final Target target;

    public AggregationAsSummaryMigration() {
        this.source = new Source();
        this.target = new Target();
        this.target.metabase.forceUpdateClusterMetaData().join();
    }

    public static void main(String[] args) {
        try (AggregationAsSummaryMigration migration = new AggregationAsSummaryMigration()) {
            var selectors = Selectors.parse("project='solomon', service='stockpile', cluster='storage_sas', bin=-, host!='cluster'");
            TimeUnit.SECONDS.sleep(10L);
            migration.migrate(selectors).join();
        } catch (Exception e) {
            e.printStackTrace();
        }
        System.exit(0);
    }

    private CompletableFuture<Void> migrate(Selectors root) {
        System.out.println("Resolve unique metrics by selector: " + root);
        return source.uniqueLabels(root, Set.of("host", "metric"))
            .thenCompose(setOfLabels -> {
                CompletableFuture<Void> future = completedFuture(null);
                for (var labels : setOfLabels) {
                    future = future.thenCompose(ignore -> migrate(root, labels));
                }
                return future;
            });
    }

    private CompletableFuture<Void> migrate(Selectors root, Labels labels) {
        Selectors selector = root.toBuilder().addOverride(Selectors.of(labels)).build();
        System.out.println("Process metrics group: " + selector);
        return source.find(selector)
            .thenCompose(keys -> {
                var it = keys.iterator();

                CompletableFuture<Void> future = completedFuture(null);
                while (it.hasNext()) {
                    var entry = it.next();
                    future = future.thenCompose(ignore -> {
                        return migrate(entry);
                    });
                }
                return future;
            });
    }

    private CompletableFuture<Void> migrate(MetricKey key) {
        System.out.println("Migrate metric: " + key);
        var labels = key.getLabels();
        if (!labels.hasKey("host")) {
            return CompletableFuture.completedFuture(null);
        }
        return target.resolveOrCreateMetric(labels.add("host", "cluster"))
            .thenCompose(targetKey -> source.loadMetrics(key)
                .thenCompose(metric -> convertMetrics(targetKey, metric)));
    }

    private CompletableFuture<Void> convertMetrics(MetricKey key, Metric<MetricKey> result) {
        var source = Objects.requireNonNull(result.getTimeseries());
        List<CompletableFuture<Void>> futures = new ArrayList<>();
        final int mask = StockpileColumn.TS.mask() | StockpileColumn.DSUMMARY.mask() | StockpileColumn.COUNT.mask() | StockpileColumn.MERGE.mask();
        MetricArchiveMutable content = new MetricArchiveMutable();
        content.setOwnerProjectId(EProjectId.SOLOMON.getNumber());
        content.setOwnerShardId(TARGET_NUM_ID);
        content.setType(MetricType.DSUMMARY);
        content.ensureCapacity(mask, Math.min(CHUNK_BYTES_SIZE, source.elapsedBytes()));

        var summary = SummaryDouble.newInstance();
        var point = RecyclableAggrPoint.newInstance();
        var it = source.iterator();
        while (it.next(point)) {
            var value = point.valueNum;
            summary.setLast(value);
            summary.setMax(value);
            summary.setMin(value);
            summary.setSum(value);
            summary.setCount(1);

            point.merge = true;
            point.count = 1;
            point.summaryDouble = summary;
            content.addRecordData(mask, point);

            if (content.bytesCount() > CHUNK_BYTES_SIZE) {
                futures.add(target.write(key, content.toImmutableNoCopy()));
                content = new MetricArchiveMutable();
                content.setType(MetricType.DSUMMARY);
                content.ensureCapacity(mask, Math.min(CHUNK_BYTES_SIZE, source.elapsedBytes()));
            }
        }
        point.recycle();

        if (content.getRecordCount() > 0) {
            futures.add(target.write(key, content.toImmutableNoCopy()));
        }

        return CompletableFutures.allOfVoid(futures);
    }

    static StockpileClient createStockpileClient(SolomonCluster cluster) {
        List<HostAndPort> addresses = cluster.addressesStockpileGrpc();

        var options = StockpileClientOptions.newBuilder(
                DefaultClientOptions.newBuilder()
                    .setRequestTimeOut(10, TimeUnit.MINUTES)
                    .setMaxInboundMessageSizeInBytes(500 << 20)) // 500 Mib
            .setExpireClusterMetadata(30, TimeUnit.SECONDS)
            .setRetryStopStrategy(StopStrategies.stopAfterAttempt(1000))
            .build();

        return StockpileClients.create(addresses, options);
    }

    static MetabaseClient createMetabaseClient(SolomonCluster cluster) {
        List<HostAndPort> addresses = cluster.hosts()
            .stream()
            .map(s -> HostAndPort.fromParts(s, SolomonPorts.COREMON_GRPC))
            .collect(toList());

        var options = MetabaseClientOptions.newBuilder(
                DefaultClientOptions.newBuilder()
                    .setRequestTimeOut(1, TimeUnit.MINUTES))
            .setExpireClusterMetadata(30, TimeUnit.MINUTES)
            .setMetaDataRequestTimeOut(5, TimeUnit.MINUTES)
            .build();

        return MetabaseClients.create(addresses, options);
    }

    @Override
    public void close() {
        source.close();
        target.close();
    }

    private static class Source implements AutoCloseable {
        private final MetricsClient metricsClient;

        Source() {
            final DcMetricsClient sas;
            {
                var metabase = createMetabaseClient(SolomonCluster.PROD_FETCHER_SAS);
                var stockpile = createStockpileClient(SolomonCluster.PROD_STORAGE_SAS);
                sas = new DcMetricsClient("sas", metabase, stockpile);
            }
            final DcMetricsClient vla;
            {
                var metabase = createMetabaseClient(SolomonCluster.PROD_FETCHER_VLA);
                var stockpile = createStockpileClient(SolomonCluster.PROD_STORAGE_VLA);
                vla = new DcMetricsClient("vla", metabase, stockpile);
            }
            metricsClient = new CrossDcMetricsClient(Map.of("sas", sas, "vla", vla));
        }

        CompletableFuture<Set<Labels>> uniqueLabels(Selectors selectors, Set<String> labels) {
            return metricsClient.uniqueLabels(UniqueLabelsRequest.newBuilder()
                .setSelectors(selectors)
                .setLabels(labels)
                .build())
                .thenApply(response -> {
                    if (!response.isOk()) {
                        throw new MetabaseClientException(response.getStatus());
                    }

                    return response.getUniqueLabels();
                });
        }

        CompletableFuture<List<MetricKey>> find(Selectors selectors) {
            return metricsClient.find(FindRequest.newBuilder()
                .setSelectors(selectors)
                .build())
                .thenApply(response -> {
                    if (!response.isOk()) {
                        throw new MetabaseClientException(response.getStatus());
                    }

                    return response.getMetrics();
                });
        }

        CompletableFuture<Metric<MetricKey>> loadMetrics(MetricKey key) {
            return metricsClient.readMany(ReadManyRequest.newBuilder()
                .addKey(key)
                .build())
                .thenApply(response -> {
                    if (!response.isOk()) {
                        throw new StockpileClientException(response.getStatus());
                    }

                    return Iterables.getOnlyElement(response.getMetrics());
                });
        }

        @Override
        public void close() {
            metricsClient.close();
        }
    }

    private static class Target implements AutoCloseable {
        private final Labels shard = Labels.of("project", "solomon", "cluster", "storage_sas", "service", "stockpile_summary");
        private final MetabaseClient metabase;
        private final StockpileClient stockpile;
        private final StockpileShardWriters writers;

        public Target() {
            this.metabase = createMetabaseClient(SolomonCluster.PRESTABLE_FETCHER);
            this.stockpile = createStockpileClient(SolomonCluster.PRESTABLE_STORAGE);
            this.writers = new StockpileShardWriters(stockpile, ForkJoinPool.commonPool());
        }

        private CompletableFuture<MetricKey> resolveOrCreateMetric(Labels labels) {
            var key = labels.addAll(shard);
            return metabase.resolveOne(ResolveOneRequest.newBuilder()
                .addAllLabels(LabelConverter.labelsToProtoList(key))
                .build())
                .thenCompose(response -> {
                    if (response.getStatus() == EMetabaseStatusCode.NOT_FOUND) {
                        return createMetric(labels);
                    }

                    if (response.getStatus() != EMetabaseStatusCode.OK) {
                        throw new MetabaseClientException(MetabaseStatus.fromCode(response.getStatus(), response.getStatusMessage()));
                    }

                    return completedFuture(toKey(response.getMetric()));
                });
        }

        private CompletableFuture<MetricKey> createMetric(Labels labels) {
            var key = labels.addAll(shard);
            return metabase.createOne(CreateOneRequest.newBuilder()
                .setMetric(ru.yandex.solomon.metabase.api.protobuf.Metric.newBuilder()
                    .setType(MetricType.DSUMMARY)
                    .addAllLabels(LabelConverter.labelsToProtoList(key))
                    .setCreatedAtMillis(System.currentTimeMillis())
                    .build())
                .build())
                .thenApply(response -> {
                    if (response.getStatus() != EMetabaseStatusCode.OK) {
                        var status = MetabaseStatus.fromCode(response.getStatus(), response.getStatusMessage());
                        throw new MetabaseClientException(status);
                    }

                    return toKey(response.getMetric());
                });
        }

        private CompletableFuture<Void> write(MetricKey key, MetricArchiveImmutable archive) {
            var stockpileKey = key.getStockpileKeys().get(0);
            return writers.write(stockpileKey.getShardId(), stockpileKey.getLocalId(), archive);
        }

        private MetricKey toKey(ru.yandex.solomon.metabase.api.protobuf.Metric metric) {
            var proto = metric.getMetricId();
            var stockpileKey = new StockpileKey("target", proto.getShardId(), proto.getLocalId());
            var kind = MetricTypeConverter.fromProto(metric.getType());
            var labels = LabelConverter.protoToLabels(metric.getLabelsList());
            return new MetricKey(kind, labels, stockpileKey);
        }

        @Override
        public void close() {
            writers.complete();
            writers.doneFuture().join();
            stockpile.close();
            metabase.close();
        }
    }
}
