package ru.yandex.solomon.experiments.gordiychuk;

import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Level;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.TearDown;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.bits.HeapBitBuf;

/**
 * @author Vladimir Gordiychuk
 */
@Fork(value = 1)
@Measurement(iterations = 20, batchSize = 30)
@Warmup(iterations = 10, batchSize = 30)
@State(Scope.Thread)
@Threads(1) //current test not support concurrent execution
@BenchmarkMode(Mode.AverageTime)
@OutputTimeUnit(TimeUnit.NANOSECONDS)
public abstract class BitBufJmh {

    /*

Benchmark                        Mode  Cnt    Score   Error  Units
BitBufJmh.Array.write            avgt   20  140.882 ± 2.782  ns/op
BitBufJmh.DirectByteBuf.write    avgt   20  131.457 ± 2.682  ns/op
BitBufJmh.HeapBitBufBench.write  avgt   20  312.471 ± 6.809  ns/op
BitBufJmh.HeapByteBuf.write      avgt   20  143.238 ± 2.922  ns/op

     */

    private int value;

    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
            .include(BitBufJmh.class.getName())
//            .include("BitBufJmh.HeapBitBufBench.write")
            .detectJvmArgs()
            .jvmArgsAppend("-Dio.netty.buffer.checkBounds=true")
//            .addProfiler(AsyncProfiler.class)
            .build();

        new Runner(opt).run();
    }

    @Setup(Level.Iteration)
    public void setUp() {
        value = ThreadLocalRandom.current().nextInt();
    }

    @Benchmark
    public Object write() {
        return write(value);
    }

    protected abstract Object write(int value);

    public static class Array extends BitBufJmh {
        private byte[] array;
        private int arrayIndex = 0;

        @Setup
        public void prepareArray() {
            array = new byte[1024 << 20]; // 1024 Mib
        }

        @Setup(Level.Iteration)
        public void resetIndex() {
            arrayIndex = 0;
        }

        @Override
        protected Object write(int value) {
            array[arrayIndex] = (byte) (value >>> 24);
            array[arrayIndex + 1] = (byte) (value >>> 16);
            array[arrayIndex + 2] = (byte) (value >>> 8);
            array[arrayIndex + 3] = (byte) value;
            arrayIndex += 4;
            return array;
        }
    }

    public static class DirectByteBuf extends BitBufJmh {
        private ByteBuf byteBuf;

        @Setup
        public void prepare() {
            byteBuf = ByteBufAllocator.DEFAULT.directBuffer(1024 << 20); // 1024 Mib
        }

        @TearDown
        public void release() {
            byteBuf.release();
        }

        @Setup(Level.Iteration)
        public void resetIndex() {
            byteBuf.resetWriterIndex();
        }

        @Override
        protected Object write(int value) {
            return byteBuf.writeInt(value);
        }
    }

    public static class HeapByteBuf extends BitBufJmh {
        private ByteBuf byteBuf;

        @Setup
        public void prepare() {
            byteBuf = ByteBufAllocator.DEFAULT.heapBuffer(1024 << 20); // 1024 Mib
        }

        @TearDown
        public void release() {
            byteBuf.release();
        }

        @Setup(Level.Iteration)
        public void resetIndex() {
            byteBuf.resetWriterIndex();
        }

        @Override
        protected Object write(int value) {
            return byteBuf.writeInt(value);
        }
    }

    public static class HeapBitBufBench extends BitBufJmh {
        private BitBuf bitBuf;

        @Setup
        public void prepare() {
            bitBuf = new HeapBitBuf(new byte[1024 << 20], 0); // 1024 Mib
        }

        @Setup(Level.Iteration)
        public void resetIndex() {
            bitBuf.resetWriterIndex();
        }

        @Override
        protected Object write(int value) {
            bitBuf.write32Bits(value);
            return bitBuf;
        }
    }
}
