package ru.yandex.solomon.experiments.gordiychuk;

import java.util.concurrent.TimeUnit;

import io.netty.buffer.UnpooledByteBufAllocator;
import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Level;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.bits.HeapBitBuf;
import ru.yandex.solomon.codec.bits.NettyBitBuf;

/**
 * @author Vladimir Gordiychuk
 */
@Fork(value = 1)
@Measurement(iterations = 20, batchSize = 30)
@Warmup(iterations = 10, batchSize = 30)
@State(Scope.Thread)
@Threads(1) //current test not support concurrent execution
@BenchmarkMode(Mode.AverageTime)
@OutputTimeUnit(TimeUnit.NANOSECONDS)
public class BitBufMethodReadJmh {
    @Param({"old", "new"})
    private String version;

    /*

Benchmark                              (version)  Mode  Cnt    Score    Error  Units
BitBufMethodReadJmh.read32Bits               old  avgt   20  273.680 ± 64.440  ns/op
BitBufMethodReadJmh.read32Bits               new  avgt   20  247.409 ±  2.632  ns/op
BitBufMethodReadJmh.read8Bits                old  avgt   20  159.422 ±  2.686  ns/op
BitBufMethodReadJmh.read8Bits                new  avgt   20  228.599 ± 20.853  ns/op
BitBufMethodReadJmh.readBit                  old  avgt   20  141.959 ±  2.583  ns/op
BitBufMethodReadJmh.readBit                  new  avgt   20  194.077 ±  4.211  ns/op
BitBufMethodReadJmh.readBits32Count7         old  avgt   20  222.328 ±  2.063  ns/op
BitBufMethodReadJmh.readBits32Count7         new  avgt   20  460.894 ± 44.463  ns/op
BitBufMethodReadJmh.readBits64Count18        old  avgt   20  373.833 ± 27.978  ns/op
BitBufMethodReadJmh.readBits64Count18        new  avgt   20  427.862 ± 35.824  ns/op
BitBufMethodReadJmh.readBits8Count3          old  avgt   20  197.426 ±  4.316  ns/op
BitBufMethodReadJmh.readBits8Count3          new  avgt   20  326.399 ±  6.894  ns/op
BitBufMethodReadJmh.readIntVarint1N          old  avgt   20  141.041 ±  2.482  ns/op
BitBufMethodReadJmh.readIntVarint1N          new  avgt   20  191.661 ± 12.061  ns/op

     */

    private BitBuf bitBuf;

    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
            .include(BitBufMethodReadJmh.class.getName())
            .detectJvmArgs()
//            .jvmArgsAppend("-Dio.netty.buffer.checkBounds=true")
            .addProfiler(AsyncProfiler.class)
            .build();

        new Runner(opt).run();
    }

    @Setup
    public void prepare() {
        int capacity = 1024 << 20;
        switch (version) {
            case "old":
                bitBuf = new HeapBitBuf(new byte[capacity], (long) capacity * Byte.SIZE);
                break;
            case "new":
                bitBuf = new NettyBitBuf(UnpooledByteBufAllocator.DEFAULT.heapBuffer(capacity), (long) capacity * Byte.SIZE);
                break;
            default:
                throw new UnsupportedOperationException("unsupported version: " + version);
        }
    }

    @Setup(Level.Iteration)
    public void resetIndex() {
        bitBuf.resetReadIndex();
        bitBuf.readBit();
    }

    @Benchmark
    public boolean readBit() {
        return bitBuf.readBit();
    }

    @Benchmark
    public byte read8Bits() {
        return bitBuf.read8Bits();
    }

    @Benchmark
    public int read32Bits() {
        return bitBuf.read32Bits();
    }

    @Benchmark
    public int readBits8Count3() {
        return bitBuf.readBitsToInt(3);
    }

    @Benchmark
    public int readBits32Count7() {
        return bitBuf.readBitsToInt(7);
    }

    @Benchmark
    public long readBits64Count18() {
        return bitBuf.readBitsToLong(18);
    }

    @Benchmark
    public int readIntVarint1N() {
        return bitBuf.readIntVarint1N(8);
    }

}
