package ru.yandex.solomon.experiments.gordiychuk;

import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import io.netty.buffer.UnpooledByteBufAllocator;
import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Level;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.bits.HeapBitBuf;
import ru.yandex.solomon.codec.bits.NettyBitBuf;

/**
 * @author Vladimir Gordiychuk
 */
@Fork(value = 1)
@Measurement(iterations = 20, batchSize = 30)
@Warmup(iterations = 10, batchSize = 30)
@State(Scope.Thread)
@Threads(1) //current test not support concurrent execution
@BenchmarkMode(Mode.AverageTime)
@OutputTimeUnit(TimeUnit.NANOSECONDS)
public class BitBufMethodWriteJmh {
    @Param({"old", "new"})
    private String version;

    /*

Benchmark                           (version)  Mode  Cnt     Score     Error  Units
BitBufMethodWriteJmh.write32Bits               old  avgt   20   269.424 ±  25.439  ns/op
BitBufMethodWriteJmh.write32Bits               new  avgt   20   308.356 ±  43.162  ns/op
BitBufMethodWriteJmh.write8Bits                old  avgt   20   233.013 ±   9.139  ns/op
BitBufMethodWriteJmh.write8Bits                new  avgt   20   330.828 ±  85.362  ns/op
BitBufMethodWriteJmh.writeBit                  old  avgt   20   157.947 ±  38.312  ns/op
BitBufMethodWriteJmh.writeBit                  new  avgt   20   202.387 ±   3.141  ns/op
BitBufMethodWriteJmh.writeBits32Count7         old  avgt   20   244.812 ±   5.563  ns/op
BitBufMethodWriteJmh.writeBits32Count7         new  avgt   20   391.862 ±   4.494  ns/op
BitBufMethodWriteJmh.writeBits64Count18        old  avgt   20   332.055 ±  34.613  ns/op
BitBufMethodWriteJmh.writeBits64Count18        new  avgt   20   338.521 ±  17.951  ns/op
BitBufMethodWriteJmh.writeBits8Count3          old  avgt   20   233.027 ±   3.968  ns/op
BitBufMethodWriteJmh.writeBits8Count3          new  avgt   20   571.058 ±  10.496  ns/op
BitBufMethodWriteJmh.writeIntVarint8           old  avgt   20   930.587 ±  14.036  ns/op
BitBufMethodWriteJmh.writeIntVarint8           new  avgt   20  1194.172 ± 223.493  ns/op
BitBufMethodWriteJmh.writeLongVarint8          old  avgt   20  1677.985 ±  36.120  ns/op
BitBufMethodWriteJmh.writeLongVarint8          new  avgt   20  1915.798 ± 334.154  ns/op

     */

    private boolean bit;
    private byte bits8;
    private byte bits8Cnt3;
    private int bits32;
    private int bits32Cnt7;
    private long bits64;
    private long bits64Cnt18;
    private BitBuf bitBuf;

    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
            .include(BitBufMethodWriteJmh.class.getName())
            .detectJvmArgs()
//            .jvmArgsAppend("-Dio.netty.buffer.checkBounds=true")
//            .addProfiler(AsyncProfiler.class)
            .build();

        new Runner(opt).run();
    }

    @Setup
    public void prepare() {
        int capacity = 1024 << 20;
        switch (version) {
            case "old":
                bitBuf = new HeapBitBuf(new byte[capacity], 0);
                break;
            case "new":
                bitBuf = new NettyBitBuf(UnpooledByteBufAllocator.DEFAULT.heapBuffer(capacity), 0);
                break;
            default:
                throw new UnsupportedOperationException("unsupported version: " + version);
        }

        bit = ThreadLocalRandom.current().nextBoolean();
        bits8 = (byte) ThreadLocalRandom.current().nextInt(255);
        bits8Cnt3 = (byte) (Math.abs(ThreadLocalRandom.current().nextInt(1 << 3)));
        bits32Cnt7 = Math.abs(ThreadLocalRandom.current().nextInt(1 << 7));
        bits64Cnt18 = Math.abs(ThreadLocalRandom.current().nextLong(1 << 18));
        bits32 = ThreadLocalRandom.current().nextInt();
        bits64 = ThreadLocalRandom.current().nextLong();
    }

    @Setup(Level.Iteration)
    public void resetIndex() {
        bitBuf.resetWriterIndex();
        bitBuf.writeBit(true);
    }

    @Benchmark
    public Object writeBit() {
        bitBuf.writeBit(bit);
        return bitBuf;
    }

    @Benchmark
    public Object write8Bits() {
        bitBuf.write8Bits(bits8);
        return bitBuf;
    }

    @Benchmark
    public Object write32Bits() {
        bitBuf.write32Bits(bits32);
        return bitBuf;
    }

    @Benchmark
    public Object writeBits8Count3() {
        bitBuf.writeBits(bits8Cnt3, 3);
        return bitBuf;
    }

    @Benchmark
    public Object writeBits32Count7() {
        bitBuf.writeBits(bits32Cnt7, 7);
        return bitBuf;
    }

    @Benchmark
    public Object writeBits64Count18() {
        bitBuf.writeBits(bits64Cnt18, 18);
        return bitBuf;
    }

    @Benchmark
    public Object writeIntVarint8() {
        bitBuf.writeIntVarint8(bits32);
        return bitBuf;
    }

    @Benchmark
    public Object writeLongVarint8() {
        bitBuf.writeLongVarint8(bits64);
        return bitBuf;
    }
}
