package ru.yandex.solomon.experiments.gordiychuk;

import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.OptionalLong;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.ints.IntSet;

import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;
import ru.yandex.solomon.coremon.meta.db.MetricsDao;
import ru.yandex.solomon.coremon.meta.db.ydb.YdbMetricsDaoFactory;
import ru.yandex.solomon.labels.intern.InterningLabelAllocator;
import ru.yandex.solomon.math.operation.reduce.CombineIterator;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.metrics.client.StockpileClientException;
import ru.yandex.solomon.metrics.client.StockpileStatus;
import ru.yandex.solomon.metrics.client.TimeSeriesCodec;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.model.timeseries.aggregation.collectors.DoublePointCollectors;
import ru.yandex.solomon.tool.StockpileHelper;
import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.tool.stockpile.StockpileShardWriters;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TReadRequest;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.StockpileClientOptions;
import ru.yandex.stockpile.client.StopStrategies;
import ru.yandex.stockpile.server.shard.load.Async;

/**
 * @author Vladimir Gordiychuk
 */
public class CloudAIAggregates {
    private static final long FROM_MILLIS = Instant.parse("2015-12-31T00:00:00Z").toEpochMilli();
    private static final long TO_MILLIS = Instant.parse("2019-12-31T00:00:00Z").toEpochMilli();

    private static YdbClient ydb;
    private static MetricsDao dao;
    private static StockpileClient stockpile;
    private static StockpileShardWriters writers;

    public static void main(String[] args) {
        var cluster = cluster(args[0]);
        ydb = YdbHelper.createYdbClient(cluster);
        dao = YdbMetricsDaoFactory.forReadOnly(ydb.table, cluster.kikimrRootPath() + "/Solomon/Coremon/V1")
            .create(-824368967, new InterningLabelAllocator());
        stockpile = stockpileClient(cluster);
        writers = new StockpileShardWriters(stockpile, ForkJoinPool.commonPool());

        try {
            var metrics = loadMetricsSync(dao);
            var mapping = prepareMapping(metrics);

            var it = mapping.entrySet().iterator();
            AtomicInteger completeCnt = new AtomicInteger();
            Async.forEach(() -> {
                if (!it.hasNext()) {
                    return CompletableFuture.completedFuture(null);
                }
                var entry = it.next();
                var key = entry.getKey();
                var value = entry.getValue();
                return recalculate(metrics, key, value)
                    .thenApply(ignore -> key);
            }, labels -> {
                System.out.println(labels + " success");
                double progress = completeCnt.incrementAndGet() * 100. / mapping.size();
                System.out.println("Progress " + String.format("%.2f%%", progress));
            }).join();

            writers.complete();
            writers.doneFuture().join();
            System.out.println("Done!");
            stockpile.close();
            ydb.close();
            metrics.release();
            System.exit(0);
        } catch (Throwable e) {
            e.printStackTrace();
            System.exit(1);
        }
    }

    private static CompletableFuture<Void> recalculate(CoremonMetricArray metrics, Labels labels, Aggregate aggregate) {
        if (aggregate.targetIdx == -1 || aggregate.sourceIdx.isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }

        return CompletableFuture.completedFuture(null)
            .thenAccept(ignore -> System.out.println(metrics.getLabels(aggregate.targetIdx) + " started..."))
            .thenCompose(ignore -> loadMetrics(metrics, aggregate))
            .thenApply(sources -> {
                int mask = StockpileColumn.TS.mask()
                    | StockpileColumn.VALUE.mask()
                    | StockpileColumn.STEP.mask()
                    | StockpileColumn.COUNT.mask()
                    | StockpileColumn.MERGE.mask();

                var filtered = sources.stream()
                    .filter(it -> !it.hasColumn(StockpileColumn.MERGE) && !it.isEmpty())
                    .map(AggrGraphDataIterable::iterator)
                    .collect(Collectors.toList());

                var it = CombineIterator.of(mask, filtered, DoublePointCollectors.ofDouble(Aggregation.SUM));
                var point = RecyclableAggrPoint.newInstance();
                var archive = new MetricArchiveMutable();
                while (it.next(point)) {
                    point.merge = false;
                    point.stepMillis = TimeUnit.MINUTES.toMillis(5);
                    archive.addRecordData(mask, point);
                }
                System.out.println(metrics.getLabels(aggregate.targetIdx) + " combined " + filtered.size());
                return archive.toImmutableNoCopy();
            })
            .thenCompose(archive -> {
                if (archive.isEmpty()) {
                    return CompletableFuture.completedFuture(null);
                }

                var shardId = metrics.getShardId(aggregate.targetIdx);
                var localId = metrics.getLocalId(aggregate.targetIdx);
                return writers.write(shardId, localId, archive);
            });
    }

    private static CompletableFuture<List<AggrGraphDataIterable>> loadMetrics(CoremonMetricArray metrics, Aggregate aggregate) {
        List<CompletableFuture<AggrGraphDataIterable>> futures = new ArrayList<>(aggregate.sourceIdx.size());
        var it = aggregate.sourceIdx.iterator();
        while (it.hasNext()) {
            int idx = it.nextInt();
            var future = stockpile.readCompressedOne(TReadRequest.newBuilder()
                .setMetricId(MetricId.newBuilder()
                    .setShardId(metrics.getShardId(idx))
                    .setLocalId(metrics.getLocalId(idx))
                    .build())
                .setBinaryVersion(StockpileFormat.CURRENT.getFormat())
                .setFromMillis(FROM_MILLIS)
                .setToMillis(TO_MILLIS)
                .setProducer(RequestProducer.SYSTEM)
                .build())
                .thenApply(response -> {
                    if (response.getStatus() != EStockpileStatusCode.OK) {
                        throw new StockpileClientException(new StockpileStatus(response.getStatus(), response.getStatusMessage()));
                    }

                    var archive = (MetricArchiveImmutable) TimeSeriesCodec.sequenceDecode(response);
                    if (archive.isEmpty()) {
                        return (AggrGraphDataIterable) archive;
                    }

                    if (archive.getType() != MetricType.DGAUGE) {
                        var mutable = archive.toMutable();
                        mutable.setType(MetricType.DGAUGE);
                        return (AggrGraphDataIterable) mutable.toImmutableNoCopy();
                    }

                    return archive;
                });
            futures.add(future);
        }
        return CompletableFutures.allOf(futures)
            .thenApply(l -> {
                System.out.println(metrics.getLabels(aggregate.targetIdx) + " loaded " + l.size());
                return l;
            });
    }

    private static Map<Labels, Aggregate> prepareMapping(CoremonMetricArray metrics) {
        var result = new HashMap<Labels, Aggregate>();
        var aggrLabel = Labels.allocator.alloc("host", "all");
        for (int index = 0; index < metrics.size(); index++) {
            var labels = metrics.getLabels(index);
            var targetLabels = labels.add(aggrLabel);
            var aggr = result.computeIfAbsent(targetLabels, l -> new Aggregate());
            if (labels.equals(targetLabels)) {
                Preconditions.checkArgument(aggr.targetIdx == -1);
                aggr.targetIdx = index;
            } else {
                aggr.sourceIdx.add(index);
            }
        }
        return result;
    }

    private static CoremonMetricArray loadMetricsSync(MetricsDao dao) throws InterruptedException {
        while (true) {
            try {
                var result = loadMetrics(dao).join();
                System.out.println("Loaded metrics: " + DataSize.shortString(result.size()));
                return result;
            } catch (Throwable e) {
                e.printStackTrace();
                TimeUnit.SECONDS.sleep(30);
            }
        }
    }

    private static CompletableFuture<CoremonMetricArray> loadMetrics(MetricsDao dao) {
        CoremonMetricArray metrics = new CoremonMetricArray();
        return dao.findMetrics(chunk -> {
                metrics.addAll(chunk);
                System.out.println("Load: " + metrics.size());
            }, OptionalLong.empty())
            .thenApply(ignore -> metrics)
            .whenComplete((ignore, e) -> {
                if (e != null) {
                    metrics.close();
                }
            });
    }

    private static StockpileClient stockpileClient(SolomonCluster cluster) {
        var opts = StockpileClientOptions.newBuilder(
                DefaultClientOptions.newBuilder()
                    .setMaxOutboundMessageSizeInBytes(256 << 20) // 256 MiB
                    .setMaxInboundMessageSizeInBytes(256 << 20) // 256 MiB
                    .setRpcExecutor(ForkJoinPool.commonPool())
                    .setResponseHandlerExecutorService(ForkJoinPool.commonPool()))
            .setRetryStopStrategy(StopStrategies.neverStop())
            .build();
        return StockpileHelper.createGrpcClient(cluster, opts);
    }

    private static SolomonCluster cluster(String name) {
        if ("sas".equals(name)) {
            return SolomonCluster.CLOUD_PROD_STORAGE_SAS;
        }

        if ("vla".equals(name)) {
            return SolomonCluster.CLOUD_PROD_STORAGE_VLA;
        }

        throw new UnsupportedOperationException("Unknown cluster: " + name);
    }

    private static class Aggregate {
        private int targetIdx = -1;
        private IntSet sourceIdx = new IntOpenHashSet();
    }
}
